/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: reporter.c,v 1.6 2002/10/09 20:56:09 brbarret Exp $
 *
 * Simple routines to print and store errors for the LAM test suite
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>


/*
 * Global variables
 */
extern char* separator;


/*
 * Local functions
 */
static int report(char *filename);
static void parse_line(char *line, char **mode, int *rank, 
		       char **filename, int *lineno);



int
main(int argc, char* argv[])
{
  if (argc != 2) {
    fprintf(stderr, "Usage: %s report_filename\n", argv[0]);
    return 1;
  }

  printf("LAM/MPI test suite reporter:\n");
  return report(argv[1]);
}


static int
report(char *basename)
{
  FILE *fp;
  long pos;
  char line[1024];
  char *mode, *filename;
  int rank, lineno;
  int len = strlen(separator);
  int last_rank = -1;
  char *last_filename = 0;
  int err_count = 0;
  int file_count = 0;

  fp = fopen(basename, "r");
  if (fp == NULL) {
    printf("File \"%s\" does not exist\n", basename);
    return 1;
  }

  fseek(fp, 0, SEEK_END);
  pos = ftell(fp);
  if (pos > 0L) {
    fseek(fp, 0, SEEK_SET);

    /* Read in the errors */

    while (fgets(line, 1024, fp) != NULL) {
      err_count++;
      parse_line(line, &mode, &rank, &filename, &lineno);

      /* Do we need new headings? */

      if (last_filename == 0 || strcmp(last_filename, filename) != 0) {
	file_count++;
	printf("==========================================================================\n");
	printf("--> File: %s\n", filename);
	last_rank = -1;
      }
      if (last_rank != rank) {
	if (last_rank != -1)
	  printf("--------------------------------------------------------------------------\n");
	printf("--> Rank: %d\n", rank);
      }
      printf("--> Line: %d\n", lineno);
      while (fgets(line, 1024, fp) != NULL) {
	if (strncmp(line, separator, len) == 0)
	  break;
	else
	  printf("%s", line);
      }

      free(mode);
      if (last_filename != NULL)
	free(last_filename);
      last_filename = filename;
      last_rank = rank;
    }
    fclose(fp);
  }
  if (last_filename != NULL)
    free(last_filename);

  printf("==========================================================================\n");
  printf("--> Total errors: %d\n", err_count);
  printf("--> Total file count: %d\n", file_count);
  if (err_count > 0)
    printf("--> Report file preserved: %s\n", basename);
  else {
    printf("--> Report file deleted: %s\n", basename);
    unlink(basename);
  }
  printf("==========================================================================\n");

  return err_count;
}


static void 
parse_line(char *line, char **mode, int *rank, char **filename, int *lineno)
{
  int i, start;
  int len = strlen(line);

  /* Get the mode name */

  for (start = i = 0; i <= len; i++)
    if (line[i] == ':' || line[i] == '\0') {
      line[i] = '\0';
      *mode = strdup(line);
      break;
    }
  ++i;

  /* Get the rank number */

  for (start = i; i <= len; i++)
    if (line[i] == ':' || line[i] == '\0') {
      line[i] = '\0';
      *rank = atoi(&line[start]);
      break;
    }
  ++i;

  /* Get the filename */

  for (start = i; i <= len; i++)
    if (line[i] == ':' || line[i] == '\0') {
      line[i] = '\0';
      *filename = strdup(&line[start]);
      break;
    }
  ++i;

  /* Get the line number */

  for (start = i; i <= len; i++)
    if (line[i] == ':' || line[i] == '\0') {
      line[i] = '\0';
      *lineno = atoi(&line[start]);
      break;
    }
}

