/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: lamtest_error.c,v 1.18 2003/06/02 20:47:21 brbarret Exp $
 *
 * Simple routines to print and store errors for the LAM test suite
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <mpi.h>

#include "lamtest_error.h"


/*
 * Local routines
 */
static void write_error(const char *file, int line, const char *rank, 
			const char *format, ...);
static char *f2c_string(char *str, int len);

void
my_abort(int status)
{
  MPI_Abort(MPI_COMM_WORLD, status);
}


static void
write_error(const char *file, int line, 
	    const char *rank, const char *format, ...)
{
  va_list ap;

  va_start(ap, format);
  write_error_int(my_abort, file, line, rank, ap);
  va_end(ap);
}


int
lamtest_warning(const char *file, int line, const char *format, ...)
{
  va_list ap;
  char *rank;
  int ret;

  /* Write the error to the file */

  rank = getenv("LAMRANK");
  if (rank == NULL)
    rank = "(unknown rank)";
  va_start(ap, format);
  write_error(file, line, rank, format, ap);

  /* Print the error to stderr */

  fprintf(stderr, 
	  "[**WARNING**]: LAM/MPI MPI_COMM_WORLD rank %s, file %s:%d:\n",
	  rank, file, line);
  ret = vfprintf(stderr, format, ap);
  va_end(ap);

  return ret;
}


void
lamtest_error(const char *file, int line, const char *format, ...)
{
  va_list ap;
  char *rank;

  /* Write the error to the file */

  rank = getenv("LAMRANK");
  if (rank == NULL)
    rank = "(unknown rank)";
  if (file == NULL)
    file = getenv("LAM_MPI_TEST_RUNNING");
  va_start(ap, format);
  write_error(file, line, rank, format, ap);

  /* Print the error to stderr */

  if (file != NULL && line != -1)
    fprintf(stderr, 
            "[**ERROR**]: LAM/MPI MPI_COMM_WORLD rank %s, file %s:%d:\n",
            rank, file, line);
  else if (file != NULL)
    fprintf(stderr, 
            "[**ERROR**]: LAM/MPI MPI_COMM_WORLD rank %s, file %s:\n",
            rank, file);
  else
    fprintf(stderr, "[**ERROR**]: LAM/MPI MPI_COMM_WORLD rank %s:\n", rank);
  vfprintf(stderr, format, ap);
  va_end(ap);

  /* Now MPI abort */

#if LAM_MPI
  /* See the man page for LAM's MPI_ABORT for an explanation */
  MPI_Abort(MPI_COMM_WORLD, (1 << 17) + 1);
#else
  MPI_Abort(MPI_COMM_WORLD, 1);
#endif
  abort();
}


#if LAMTEST_HAVE_MPI_F77
void
F77_FUNC_(lamtest_error_f, LAMTEST_ERROR_F)(char *msg_f, int len)
{
  char *msg = f2c_string(msg_f, len);
  lamtest_error(NULL, -1, msg);

  /* We'll never get here, but... */

  free(msg);
}

void
F77_FUNC_(lamtest_check_size_f, LAMTEST_CHECK_SIZE_F)(int* min, int* want_abort)
{
  lamtest_check_size(NULL, -1, *min, *want_abort);
}
#endif


void 
lamtest_check_size(const char *file, int line, int min, int want_abort)
{
  int size;
  MPI_Comm_size(MPI_COMM_WORLD, &size);

  if (size < min) {
    if (want_abort)
      lamtest_error(file, line, 
		    "This test requires at least %d processes to run\n", 
		    min);
    else
      lamtest_warning(file, line, "This test performs best with at least %d processes.  It will still run, but\nsome tests may be skipped and/or reduced.\n", min);
  }
}

void 
lamtest_need_even(const char *file, int line)
{
  int size;
  MPI_Comm_size(MPI_COMM_WORLD, &size);

  if (size % 2 != 0)
    lamtest_error(file, line, 
		  "This test requires an even number of processes to run.  Aborting.\n");
}


static char *
f2c_string(char *str, int len)
{
  char		*cstr;
  char		*end;
  int		i;

  /* Leading and trailing blanks are discarded. */

  end = str + len - 1;
  for (i = 0; (i < len) && (*str == ' '); ++i, ++str) 
    continue;

  if (i >= len)
    len = 0;
  else {
    for (; (end > str) && (*end == ' '); --end)
      continue;
    
    len = end - str + 1;
  }

  /* Allocate space for the C string. */

  if ((cstr = malloc(len + 2)) == NULL)
    return NULL;

  /* Copy F77 string into C string and NULL terminate it. */
  
  if (len > 0) {
    strncpy(cstr, str, len);
    strcat(cstr, "\n");
  }
  cstr[len + 1] = 0;

  return cstr;
}
