/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: send2.c,v 1.7 2003/06/28 11:07:19 jsquyres Exp $
 *
 *	Function:	- tests synchonicity of MPI_Ssend between two ranks
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mpi.h>

#include <lamtest_error.h>

#define MAX_MSG_SIZE 10485760
#define SMALL_SIZE 65537


int
main(int argc, char **argv)
{
  int size, rank, mcwrank;
  int i, len;
  int count, total;
  int left, right;
  int left_value, right_value;
  MPI_Comm comm1, comm2;
  char *send_left, *send_right;
  char *recv_left, *recv_right;
  MPI_Request req[4];
  int max_msg_size;

  MPI_Init(&argc, &argv);
  MPI_Comm_rank(MPI_COMM_WORLD, &mcwrank);
  MPI_Comm_rank(MPI_COMM_WORLD, &size);

  /* We need at least 2 to run */

  lamtest_check_size(__FILE__, __LINE__, 2, 1);

  if (mcwrank == 0) {
    printf("It is not unusual for this test to take several minutes,\n"
	   "particularly on slow networks and/or in lamd mode.\n"
	   "Please be patient.\n");
    fflush(stdout);
  }

  /* Make a new comm with a different ordering than MPI_COMM_WORLD (to
     ensure that the underlying message transport doesn't rely on MCW
     ordering) */

  if (size > 4)
    MPI_Comm_split(MPI_COMM_WORLD, mcwrank % 2, -rank, &comm1);
  else
    MPI_Comm_split(MPI_COMM_WORLD, 0, -mcwrank, &comm1);
  MPI_Comm_dup(comm1, &comm2);
  MPI_Comm_size(comm1, &size);
  MPI_Comm_rank(comm1, &rank);

  /* Look for overrides for RPI's that can't allocate much "special"
     memory */

  max_msg_size = MAX_MSG_SIZE;
  if (getenv("LAM_MPI_test_small_mem") != NULL)
    max_msg_size = SMALL_SIZE;
  if (mcwrank == 0)
    printf("NOTICE: Using max message size: %d\n", max_msg_size);

  /* Send short and long messages */

  send_left = malloc(max_msg_size * 4);
  if (send_left == NULL)
    lamtest_error(__FILE__, __LINE__, "Malloc failed");
  send_right = send_left + (1 * max_msg_size);
  recv_left = send_left + (2 * max_msg_size);
  recv_right = send_left + (3 * max_msg_size);

  /* Fill the send messages with expected values */

  memset(send_left, (rank % 254) + 1, max_msg_size * 2);

  /* Calculate my left and right neighbors */

  left = (rank + 1) % size;
  left_value = (left % 254) + 1;
  right = (rank + size - 1) % size;
  right_value = (right % 254) + 1;

  /* Do the sends */

  for (total = 0, len = 1; len <= max_msg_size; len <<= 1, ++total)
    continue;
  --total;

  lamtest_progress_start(total);
  for (count = 0, len = 1; len <= max_msg_size; len <<= 1, ++count) {
    memset(recv_left, 0, len);
    memset(recv_right, 0, len);

    MPI_Isend(send_left, len, MPI_CHAR, left, 111, comm1, &req[0]);
    MPI_Isend(send_right, len, MPI_CHAR, right, 222, comm2, &req[1]);
    MPI_Irecv(recv_left, len, MPI_CHAR, left, 222, comm2, &req[2]);
    MPI_Irecv(recv_right, len, MPI_CHAR, right, 111, comm1, &req[3]);

    MPI_Waitall(4, req, MPI_STATUSES_IGNORE);

    /* Check the received message contents */

    for (i = 0; i < len; ++i) {
      if (recv_left[i] != left_value)
        lamtest_error(__FILE__, __LINE__,
                      "ERROR: Got wrong value in received message, got recv_left[%d] = %d, should have been %d",
                      i, recv_left[i], left_value);
      if (recv_right[i] != right_value)
        lamtest_error(__FILE__, __LINE__,
                      "ERROR: Got wrong value in received message, got recv_right[%d] = %d, should have been %d",
                      i, recv_right[i], right_value);
    }

    /* Print the progress bar */

    lamtest_progress(count);
  }
  lamtest_progress_end();

  /* All done */

  free(send_left);
  MPI_Comm_free(&comm2);
  MPI_Comm_free(&comm1);
  MPI_Finalize();
  return 0;
}
