/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: spawn_rrobin_node.c,v 1.14 2003/06/19 04:37:12 jsquyres Exp $
 *
 * Program to test MPI_Comm_spawn with info arguments.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <mpi.h>
#include "lamtest_error.h"
#include "lamtest_spawn.h"


/* Local calls */


void
do_parent(char *cmd, int rank)
{
  int *errcodes, err;
  int i, start_node, size, node, nnodes;
  char *spawn_argv[3];
  MPI_Comm child_inter;
  MPI_Comm intra;
  int found;
  MPI_Info info;
  char str[256];
  MPI_Comm comm;
  int *nodes = 0;
  struct route *routes = 0;

  /* Ensure we have 3 processes */

  lamtest_check_size(__FILE__, __LINE__, 2, 1);

  spawn_argv[0] = cmd_argv1;
  spawn_argv[1] = cmd_argv2;
  spawn_argv[2] = NULL;

  MPI_Comm_size(MPI_COMM_WORLD, &size);

  /* Find out who's got cmd */

  find_cmd(&comm, cmd, 1, &found);
  if (found == 0)
    return;

  /* Make an info, set us to start on some middle node (i.e., not n0
     if we can help it, because that's the default LAM spawn
     scheduling policy) */

  MPI_Info_create(&info);
  start_node = size / 2;
  nnodes = getntype(0, NT_CAST);
  if (nnodes == -1)
    lamtest_error(__FILE__, __LINE__, 
                  "LAM internal call getntype() returned -1; did "
                  "value for NT_CAST change?\n");
  if (start_node == 0 && nnodes > 1)
    start_node = 1;
  snprintf(str, sizeof(str) - 1, "n%d", start_node);
  str[sizeof(str) - 1] = '\0';
  MPI_Info_set(info, "lam_spawn_sched_round_robin", str);
  
  /* Now try the spawn if it's found anywhere */

  errcodes = malloc(sizeof(int) * nnodes);
  memset(errcodes, -1, nnodes);
  MPI_Comm_set_errhandler(comm, MPI_ERRORS_RETURN);
  err = MPI_Comm_spawn(cmd, spawn_argv, nnodes, info, 0, comm,
		       &child_inter, errcodes); 

  for (i = 0; i < nnodes; i++)
    if (errcodes[i] != MPI_SUCCESS)
      lamtest_error(__FILE__, __LINE__, "ERROR: MPI_Comm_spawn returned "
		    "errcodes[%d] = %d\n", i, errcodes[i]);
  if (err != MPI_SUCCESS)
    lamtest_error(__FILE__, __LINE__, 
		  "ERROR: MPI_Comm_spawn returned errcode = %d\n", err);

  /* See if everyone launched on the right node.  Use some internal
     LAM function calls to verify this. */

  if (rank == 0) {
    nodes = malloc((sizeof(int) + sizeof(struct route)) * nnodes);
    if (nodes == NULL)
      lamtest_error(__FILE__, __LINE__, "ERROR: Malloc failed!");
    routes = (struct route *) (nodes + nnodes);

    /* Query the lamd to get route information */

    getnodes(nodes, nnodes, 0, NT_CAST);
    for (i = 0; i < nnodes; ++i) {
      routes[i].r_nodeid = nodes[i];
      getrent(&routes[i]);
    }

    /* Check each processes' node to ensure that it's what it is
       suppsoed to be */

    start_node = start_node % nnodes;
    for (i = 0; i < nnodes; ++i) {
      MPI_Recv(&node, 1, MPI_INT, i, node_tag, child_inter, 
	       MPI_STATUS_IGNORE);

      if (node != start_node)
	lamtest_error(__FILE__, __LINE__, 
		      "Error: spawned rank %d is running on n%d, "
		      "but should be on n%d\n", i, node, start_node);
      else if (verbose)
	printf("Spawned rank %d is on n%d\n", i, start_node);

      /* Find the next schedulable node */

      start_node = (start_node + 1) % nnodes;
      while ((routes[start_node].r_nodetype & NT_WASTE) != 0)
        start_node = (start_node + 1) % nnodes;
    }

    /* Free both memory arrays */

    free(nodes);
  }

  /* Now do a simple ping pong to everyone in the child */

  MPI_Intercomm_merge(child_inter, 0, &intra);
  all_to_all(intra);

  /* Clean up */

  MPI_Info_free(&info);
  MPI_Comm_free(&intra);
  free_inter(child_inter, 1);
  MPI_Comm_free(&comm);
  free(errcodes);
}


void
do_target(char *argv0, char *argv1, char *argv2, MPI_Comm parent)
{
  int rank, node;
  MPI_Comm intra;

  MPI_Comm_rank(MPI_COMM_WORLD, &rank);

  /* Check that we got the argv that we expected */

  if (strcmp(argv1, cmd_argv1) != 0)
    lamtest_error(__FILE__, __LINE__, "ERROR: Spawn target rank %d got argv[1]=\"%s\" when expecing \"%s\"\n",
		  rank, argv1, cmd_argv1);
  if (strcmp(argv2, cmd_argv2) != 0)
    lamtest_error(__FILE__, __LINE__, "ERROR: Spawn target rank %d got argv[2]=\"%s\" when expecing \"%s\"\n",
		  rank, argv2, cmd_argv2);

  /* Send my LAM node number to rank 0 in my parent.  Use internal LAM
     call to get our LAM node number. */

  node = getnodeid();
  if (verbose)
    printf("Spawned rank %d reporting that I'm on n%d\n", rank, node);
  MPI_Send(&node, 1, MPI_INT, 0, node_tag, parent);

  /* Now merge it down to an intra and do a simple all-to-all to
     everyone in the parent */

  MPI_Intercomm_merge(parent, 0, &intra);
  all_to_all(intra);
  MPI_Comm_free(&intra);

  free_inter(parent, 0);
}


