/*
 * Copyright (c) 2001-2002 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: spawn_appschema.c,v 1.10 2002/10/09 20:55:59 brbarret Exp $
 *
 * Program to test MPI_Comm_spawn with simple arguments.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include "mpi.h"

#include "lamtest_error.h"
#include "lamtest_spawn.h"


void
do_parent(char *cmd, int rank)
{
  int *errcode, err;
  int i;
  MPI_Comm child_inter;
  MPI_Comm intra;
  char filename[256];
  MPI_Info info;
  MPI_Comm comm;
  FILE *fp;
  int count;

  /* Ensure we have 3 processes */

  lamtest_check_size(__FILE__, __LINE__, 3, 1);

  /* Find out who's got cmd */

  MPI_Comm_dup(MPI_COMM_WORLD, &comm);
  MPI_Comm_size(comm, &count);

  /* Have the parent write out an appschema file */

  MPI_Info_create(&info);
  if (rank == 0) {
    sprintf(filename, "lam_test_appschema.%d", getpid());
    fp = fopen(filename, "w");
    if (fp == NULL)
      lamtest_error(__FILE__, __LINE__, "ERROR: Parent rank 0 could not open filename \"%s\" for writing.\n",
		    filename);

    fprintf(fp, "# This is a sample appschema that is automatically generated for the\n");
    fprintf(fp, "# LAM/MPI test suite.\n");
    fprintf(fp, "# It should be removed at the end of a successful test.\n");
    fprintf(fp, "# This simply launches %d copies of \"%s\".\n", count, cmd);
    fprintf(fp, "# We don't do any node placement here because we cannot assume\n");
    fprintf(fp, "# how many nodes the user has lambooted.\n");
    fprintf(fp, "-np %d -s h %s %s %s\n", count, cmd, cmd_argv1, cmd_argv2);
    fclose(fp);

    MPI_Info_set(info, "file", filename);
  }

  /* Spawn using the appschema, and delibrarately don't give the first
     three arguments to MPI_Comm_spawn.  This is documented behavior
     for LAM/MPI. */

  errcode = malloc(sizeof(int) * count);
  memset(errcode, 0, count * sizeof(int));
  MPI_Comm_set_errhandler(comm, MPI_ERRORS_RETURN);
  err = MPI_Comm_spawn(0, 0, 0, info, 0, comm, &child_inter, errcode);

  if (rank == 0)
    unlink(filename);

  for (i = 0; i < count; i++)
    if (errcode[i] != MPI_SUCCESS)
      lamtest_error(__FILE__, __LINE__, 
		    "ERROR: MPI_Comm_spawn returned errcode[%d] = %d\n", 
		    i, errcode[i]);
  if (err != MPI_SUCCESS)
    lamtest_error(__FILE__, __LINE__, 
		  "ERROR: MPI_Comm_spawn returned errcode = %d\n", err);

  /* Now do a simple ping pong to everyone in the child */

  MPI_Intercomm_merge(child_inter, 0, &intra);
  all_to_all(intra);
  MPI_Comm_free(&intra);
  MPI_Comm_free(&comm);

  /* Clean up */

  MPI_Info_free(&info);
  free_inter(child_inter, 1);
  free(errcode);
}


void
do_target(char *argv0, char *argv1, char *argv2, MPI_Comm parent)
{
  int rank;
  MPI_Comm intra;

  MPI_Comm_rank(MPI_COMM_WORLD, &rank);

  /* Check that we got the argv that we expected */

  if (strcmp(argv1, cmd_argv1) != 0)
    lamtest_error(__FILE__, __LINE__, "ERROR: Spawn target rank %d got argv[1]=\"%s\" when expecing \"%s\"\n",
		  rank, argv1, cmd_argv1);
  if (strcmp(argv2, cmd_argv2) != 0)
    lamtest_error(__FILE__, __LINE__, "ERROR: Spawn target rank %d got argv[2]=\"%s\" when expecing \"%s\"\n",
		  rank, argv2, cmd_argv2);

  /* Now merge it down to an intra and do a simple all-to-all to
     everyone in the parent */

  MPI_Intercomm_merge(parent, 0, &intra);
  all_to_all(intra);
  MPI_Comm_free(&intra);

  free_inter(parent, 0);
}
