/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: sizes.c,v 1.11 2003/06/23 11:24:47 jsquyres Exp $
 *
 * Send messages of all different sizes to all the other ranks.  We do
 * this to test all the different message sending protocols.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <mpi.h>

#include <lamtest_error.h>

#define local_max(a,b)	(((a) > (b)) ? (a) : (b))


/*
 * globals
 */
unsigned char *send_buffer = 0;
unsigned char *recv_buffer = 0;
int tag = 201;
int max_size = 1048576;


void 
sendrecv(int target, int me, int base, int size)
{
  int i;

  /* Sanity check the size first */

  if (size <= 0)
    return;
  if (size <= base / 2 || size >= base * 2)
    return;  

  /* Do the communication */

  memset(recv_buffer, 0, size);
  if (me == target)
    return;
  else if (me < target) {
    MPI_Send(send_buffer, size, MPI_CHAR, target, tag, MPI_COMM_WORLD);
    MPI_Recv(recv_buffer, size, MPI_CHAR, target, tag, MPI_COMM_WORLD, 
	     MPI_STATUS_IGNORE);
  } else {
    MPI_Recv(recv_buffer, size, MPI_CHAR, target, tag, MPI_COMM_WORLD, 
	     MPI_STATUS_IGNORE);
    MPI_Send(send_buffer, size, MPI_CHAR, target, tag, MPI_COMM_WORLD);
  } 

  /* Ensure that we got the message properly */

  for (i = 0; i < size; ++i) {
    if (recv_buffer[i] != (i % 255)) {
      lamtest_error(__FILE__, __LINE__, 
		    "recv_msg[%d]=%d, it should be %d\n", 
		    i, recv_buffer[i], (i % 255));
    }
  }
}


int
main(int argc, char* argv[])
{
  int i;
  int rank;
  int size;
  int len;
  int total, count;
  int max_msg_size;

  MPI_Init(&argc, &argv);
  MPI_Comm_size(MPI_COMM_WORLD, &size);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);

  lamtest_check_size(__FILE__, __LINE__, 2, 0);
  max_msg_size = max_size;

  /* If we only have one rank, we must reduce the size to be not long
     (long requires the rendevouz protocol, which won't happen in the
     logic below -- the code will deadlock with one rank and long
     messages) */

  if (size == 1)
    max_size = 1;
  else
    max_size = max_msg_size;
 
  if (rank == 0) {
    printf("It is not unusual for this test to take several minutes,\n"
	   "particularly on slow networks and/or in lamd mode.\n"
	   "Please be patient.\n");
    fflush(stdout);
  }

  /* Alloc the buffers */

  send_buffer = malloc(max_size * 2);
  recv_buffer = malloc(max_size * 2);
  for (i = 0; i < (max_size * 2); ++i)
    send_buffer[i] = (char) (i % 255);

  /* Go through all the sizes, jitter a little on both sides of the
     power of two.  Send/recv with all the other ranks for each
     size. */

  for (total = 0, len = 1; len <= max_size; len *= 2, ++total)
    continue;
  --total;

  lamtest_progress_start(total);
  for (count = 0, len = 1; len <= max_size; len *= 2, ++count) {
    for (i = 0; i < size; i++) {
      if (len * 0.8 != len)
	sendrecv(i, rank, len, len * 0.8);
      if (len * 0.9 != len)
	sendrecv(i, rank, len, len * 0.9);
      sendrecv(i, rank, len, len);
      if (len * 1.1 != len && (len * 1.1) < max_msg_size)
	sendrecv(i, rank, len, len * 1.1);
      if (len * 1.2 != len && (len * 1.2) < max_msg_size)
	sendrecv(i, rank, len, len * 1.2);
    }
    lamtest_progress(count);
  }
  lamtest_progress_end();

  MPI_Finalize();

  free(send_buffer);
  free(recv_buffer);

  return 0;
}
