/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: ring.c,v 1.8 2003/06/02 02:25:35 jsquyres Exp $
 *
 * Test some long, big messages in a ring pattern.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "mpi.h"

#include "lamtest_error.h"

#define SIZE 1048576
#define ITERATIONS 5

static void check(char *buffer, int size, int val);
int rank;


int 
main(int argc, char **argv)
{
  int num, from_real;
  int size, tag, next, from;
  char *buffer = 0;
  int bufsize = SIZE;

  /* Start up MPI */

  MPI_Init(&argc, &argv);
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  MPI_Comm_size(MPI_COMM_WORLD, &size);

  /* If we only have one rank, we must reduce the size to be not long
     (long requires the rendevouz protocol, which won't happen in the
     logic below -- the code will deadlock with one rank and long
     messages) */

  lamtest_check_size(__FILE__, __LINE__, 2, 0);
  if (size == 1)
    bufsize = 1;
 
  /* Arbitrarily choose 201 to be our tag.  Calculate the rank of the
     next process in the ring.  Use the modulus operator so that the
     last process "wraps around" to rank zero.  Use MPI_ANY_SOURCE
     just to make the test a bit more difficult.  */

  tag = 201;
  next = (rank + 1) % size;
  from = MPI_ANY_SOURCE;
  from_real = (rank - 1 + size) % size;

  buffer = (char*) malloc(bufsize);
  memset(buffer, 0, bufsize);

  /* If we are the "console" process, get a integer from the user to
     specify how many times we want to go around the ring */

  if (rank == 0) {
    num = ITERATIONS;

    buffer[0] = num % 127;
    MPI_Send(buffer, bufsize, MPI_CHAR, next, tag, MPI_COMM_WORLD); 
  }

  /* Pass the message around the ring.  The exit mechanism works as
     follows: the message (a positive integer) is passed around the
     ring.  Each time is passes rank 0, it is decremented.  When each
     processes receives the 0 message, it passes it on to the next
     process and then quits.  By passing the 0 first, every process
     gets the 0 message and can quit normally. */

  while (1) {
    memset(buffer, 0, bufsize);
    MPI_Recv(buffer, bufsize, MPI_CHAR, from, tag, MPI_COMM_WORLD, 
	     MPI_STATUS_IGNORE);
    check(buffer, bufsize, from_real % 127);
    num = buffer[0];

    if (rank == 0) {
      num--;
      buffer[0]--;
    }

    memset(buffer + 1, (rank % 127), (bufsize - 1));
    MPI_Send(buffer, bufsize, MPI_CHAR, next, tag, MPI_COMM_WORLD);

    if (num == 0)
      break;
  }

  /* Do the final recv */

  if (rank == 0)
    MPI_Recv(buffer, bufsize, MPI_INT, from, tag, MPI_COMM_WORLD, 
	     MPI_STATUS_IGNORE);

  /* Quit */

  free(buffer);
  MPI_Finalize();
  return 0;
}


static void
check(char *buffer, int size, int val)
{
  int i;

  for (i = 1; i < size; ++i)
    if (buffer[i] != val) {
      lamtest_error(__FILE__, __LINE__, 
		    "Rank %d: found value %d at pos %d when expecting value %d\n",
		    rank, buffer[i], i, val);
    }
}


