/*
 * Copyright (c) 2001-2003 The Trustees of Indiana University.  
 *                         All rights reserved.
 * Copyright (c) 1998-2001 University of Notre Dame. 
 *                         All rights reserved.
 * Copyright (c) 1994-1998 The Ohio State University.  
 *                         All rights reserved.
 * 
 * This file is part of the LAM/MPI software package.  For license
 * information, see the LICENSE file in the top level directory of the
 * LAM/MPI source distribution.
 * 
 * $HEADER$
 *
 * $Id: isendTest.c,v 1.5 2003/06/02 02:25:35 jsquyres Exp $
 *
 * Test originally contributed by Robin Humble.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <sys/time.h>

#include <mpi.h>

#define MSGS 500	       /* send approx this many messages total */
#define MAX_MSG_SIZE 5000      /* max number of ints to send in each message */
#define TAG 916

static void processIncoming(void);
static int checkSum(char *data, int size);

static int myRank, numProcs;
static int allRecvCnt, maxMsgs;

static int *inc_buf = 0;


int
main(int argc, char **argv)
{
  int i, j, sum, rank, msgsPerProc;
  int *msg[MSGS], len[MSGS], *sendCnt, allSendCnt;
  MPI_Request *r;
  int index, flag, pendingSends;

  MPI_Init(&argc, &argv);
  MPI_Comm_rank(MPI_COMM_WORLD, &myRank);
  MPI_Comm_size(MPI_COMM_WORLD, &numProcs);

  if (myRank == 0) {
    printf("It is not unusual for this test to take several minutes,\n"
	   "particularly on slow networks and/or in lamd mode.\n"
	   "Please be patient.\n");
    fflush(stdout);
  }

  msgsPerProc = MSGS / (numProcs * numProcs);
  maxMsgs = msgsPerProc * numProcs;

  sendCnt = malloc(numProcs * sizeof(int));
  for (i = 0; i < numProcs; i++)
    sendCnt[i] = 0;

  /* Alloc out an array of MPI_Request's, and zero them out */

  r = malloc(maxMsgs * sizeof(MPI_Request));
  for (i = 0; i < maxMsgs; ++i)
    r[i] = MPI_REQUEST_NULL;

  srand48(231 + myRank);

  /* generate a pile of messages and checksums */

  for (i = 0; i < msgsPerProc; i++) {
    len[i] = MAX_MSG_SIZE * drand48() + 1 /* for checksum */ ;
    msg[i] = malloc(len[i] * sizeof(int));

    /* fill message */

    for (j = 0; j < len[i] - 1; j++)
      msg[i][j] = i;

    /* find checksum */

    msg[i][len[i] - 1] =
      checkSum((char *) (msg[i]), (len[i] - 1) * sizeof(int));
  }

  allSendCnt = 0;
  allRecvCnt = 0;
  pendingSends = 0;

  /* loop while still have messages to send */

  while (allSendCnt < maxMsgs) {

    /* sometimes send to a random rank unless done too many already */

    if (drand48() > 0.5) {
      rank = numProcs * drand48();
      if (sendCnt[rank] < msgsPerProc) {
	i = sendCnt[rank];
	MPI_Isend(msg[i], len[i], MPI_INT, rank, TAG,
		  MPI_COMM_WORLD, &r[allSendCnt]);
	sendCnt[rank]++;
	allSendCnt++;
	++pendingSends;
      }
    }

    /* sometimes see if messages have arrived */

    if (drand48() > 0.7)
      processIncoming();

    /* Allow any pending sends to complete.  This is certainly not the
       best way to do this, but it's quick-n-dirty */

    if (pendingSends > 0) {
      do {
	MPI_Testany(maxMsgs, r, &index, &flag, MPI_STATUS_IGNORE);
	if (flag == 1)
	  --pendingSends;
      } while (pendingSends > 0 && flag == 1);
    }
  }
  assert(allSendCnt == maxMsgs);

  sum = 0;
  for (i = 0; i < numProcs; i++) {
    assert(sendCnt[i] == msgsPerProc);
    sum += sendCnt[i];
  }
  assert(sum == allSendCnt);

  /* wait for the remaining incoming messages */

  while (allRecvCnt < maxMsgs)
    processIncoming();

  assert((allSendCnt == maxMsgs) && (allSendCnt == maxMsgs));

  MPI_Barrier(MPI_COMM_WORLD);

  /* everyone finished now... */

  for (i = 0; i < msgsPerProc; i++)
    free(msg[i]);
  free(r);
  free(sendCnt);
  free(inc_buf);

  MPI_Finalize();

  return 0;
}


void
processIncoming(void)
{
  static int reqPosted = 0;
  static MPI_Request r;
  MPI_Status s;
  int arrived;
  int finished = 0;

  if (inc_buf == 0)
    inc_buf = malloc((MAX_MSG_SIZE + 1) * sizeof(int));

  /* loop processing messages and checking checkSums */
  while (!finished) {
    /* sometimes post IRecv and sometimes not */
    if (!reqPosted) {
      if ((drand48() > 0.5) && (allRecvCnt < maxMsgs)) {
	MPI_Irecv(inc_buf, MAX_MSG_SIZE + 1, MPI_INT, MPI_ANY_SOURCE, TAG,
		  MPI_COMM_WORLD, &r);
	reqPosted = 1;
      }
    } else {
      MPI_Test(&r, &arrived, &s);
      if (arrived) {
	int size, sum;

	MPI_Get_count(&s, MPI_INT, &size);

	/* do checkSum */
	sum = checkSum((char *) inc_buf, (size - 1) * sizeof(int));
	if (sum != inc_buf[size - 1]) {
	  printf("p%d oops - calc'd sum %d != message sum %d -- pid %d\n",
		 myRank, sum, inc_buf[size - 1], getpid());

	  MPI_Abort(MPI_COMM_WORLD, 0);
	  exit(1);
	}

	reqPosted = 0;

	allRecvCnt++;

	if (allRecvCnt >= maxMsgs)
	  finished = 1;
      } else {
	/* no messages waiting so go back... */

	finished = 1;
      }
    }

    /* randomly quit out of this loop */

    if (drand48() > 0.5)
      finished = 1;

    /* delay */

    if (drand48() > 0.5) {

      /* Use a lame-o "sleep" */

      struct timeval start, end;
      gettimeofday(&start, 0);
      while (1) {
	gettimeofday(&end, 0);
	if (end.tv_sec > start.tv_sec)
	  end.tv_usec += 1000;
	if (end.tv_usec - 1 > start.tv_usec)
	  break;
      }
    }
  }
}


/*
 * trivial checksum routine
 */
int
checkSum(char *data, int size)
{
  int i, result;

  result = 0;
  for (i = 0; i < size; i++)
    result += data[i];

  return result;
}
