/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	$Id: lamnodes.c,v 1.6 2001/03/01 02:38:28 jsquyres Exp $
 *
 *	Function:	- retrieve hostnames from LAM nx nomenclature
 */

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <args.h>
#include <dl_inet.h>
#include <portable.h>
#include <priority.h>
#include <terror.h>
#include <rreq.h>


/*
 * Local variables
 */
static char		name[1024];
static int4		nlinks;
static struct dolink	*links;
static int              fl_name;
static int              fl_lamnode;
static int              fl_showcpu;


static void
nodename(int node, int want_name)
{
    struct hostent *hent;

    if (node < 0 || node >= nlinks
	    || links[node].dol_link == NOTLINKID) {
	strcpy(name, "invalid node");
    } else {
	hent = gethostbyaddr((char *)&links[node].dol_addr.sin_addr,
			sizeof(struct in_addr), AF_INET);
	if (hent && want_name)
	    strncpy(name, hent->h_name, sizeof(name));
	else
	    sprintf(name, "%s", inet_ntoa(links[node].dol_addr.sin_addr));
    }
}


static void
print_node(int node, char *name, int ncpus)
{
  if (fl_lamnode)
    if (fl_showcpu)
      printf("n%d\t%s:%d\n", node, name, ncpus);
    else
      printf("n%d\t%s\n", node, name);
  else
    if (fl_showcpu)
      printf("%s:%d\n", name, ncpus);
    else
      printf("%s\n", name);
}


int
main(int argc, char **argv)
{
    int4 node;
    int n_index, n_flags;
    int *pnodes;
    struct route r;

    validopts("hinc");
    if (do_args(&argc, argv)) {
        show_help("lamnodes", "usage", NULL);
	exit(errno);
    }

    if (opt_taken('h')) {
        show_help("lamnodes", "usage", NULL);
	exit(0);
    }
    /* Do we want the IP number or name? */
    fl_name = !opt_taken('i');
    /* Do we want the node numbers (i.e., nx) shown? */
    fl_lamnode = !opt_taken('n');
    /* Do we want the cpu count shown? */
    fl_showcpu = !opt_taken('c');

    if (kinit(PRCMD)) {
      show_help(NULL, "no-lamd", "lamnodes", NULL);
      exit(LAM_EEXIT);
    }

    if (ldogetlinks(&links, &nlinks))
	lamfail("lamnodes (ldogetlinks)");

    if (nid_parse(&argc, argv) || (errno = (argc == 1) ? 0 : EUSAGE)) {
        show_help("lamnodes", "usage", NULL);
	kexit(errno);
    }

    pnodes = (int4 *) malloc((unsigned) (nlinks * sizeof(int4)));
    if (pnodes == 0) 
      lamfail("lamnodes (malloc)");

    if (getnodes(pnodes, nlinks, 0, NT_CAST)) 
      lamfail("lamnodes (getnodes)");

    nid_get(&n_index, &node, &n_flags);
    if (n_index < 0) {
	for (node = 0; node < nlinks; node++) {
	    nodename(node, fl_name);
	    r.r_nodeid = node;
	    getrent(&r);
	    print_node(node, name, r.r_ncpus);
	}
    } else {
	do {
	    if (node == LOCAL)
		node = getnodeid();

	    r.r_nodeid = node;
	    getrent(&r);
	    nodename(node, fl_name);
	    print_node(node, name, r.r_ncpus);

	    nid_get(&n_index, &node, &n_flags);
	} while (n_index);
    }
    free(pnodes);

    return 0;
}
