/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	$Id: xcoloroff.c,v 6.2 2000/10/26 03:20:49 jsquyres Exp $
 *
 *	Function:	- enable a color in the XMPI trace stream
 */

#include <blktype.h>
#include <mpisys.h>

/*
 *	XMPI_Coloroff
 *
 *	Function:	- drop a buoy in the trace stream
 *	Accepts:	- buoy name
 */
/*@

XMPI_Coloroff - LAM/MPI-specific function to disable a color in the
XMPI trace stream.

Notes:

The LAM implementation of MPI is integrated with the XMPI run/debug
viewer.  It can generate tracefiles and on-the-fly trace streams
suitable for display in XMPI.  

A new functionality in XMPI is the ability to enable and disable
select colors in the trace stream.  LAM/MPI allows this functionality
with the 'XMPI_Coloron' and 'XMPI_Coloroff' functions.  

'XMPI_Coloron' is called with 'red', 'green', and 'blue' parameters.
Each value may be from 0 to 255.  The resulting RGB composite will
become activated for that rank at that point in time.  Enabling and
disabling colors is a local action; the calls will return
immediately.  The color will be activated or deactivated on the
timeline corresponding to the rank that invoked 'XMPI_Coloron' /
'XMPI_Colorff' in the XMPI trace window.

Only one color is active at a time.  However, 'XMPI_Coloron' may be
invoked multiple times to build a stack of colors.  'XMPI_Coloroff'
will pop the last color off the stack and make the previous color
active.

If this function is invoked and tracing is not active, the color
deactivation is ignored.

There is no profiling version of this function.

This is a LAM/MPI-specific function and is intended mainly for use
with XMPI.  If this function is used, it should be used in conjunction
with the 'LAM_MPI' C preprocessor macro

.vb
  #if LAM_MPI
     XMPI_Coloroff();
  #endif
.ve

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_INTERN

.seealso: XMPI_Buoy, XMPI_Coloron
@*/
int 
XMPI_Coloroff()
{
  lam_initerr();
  lam_setfunc(BLKXMPICOLOROFF);

  /* Disable the last color in the trace stream */

  if (LAM_TRACE_TOP())
    if (lam_tr_coloroff(MPI_Wtime()) != 0)
      return(lam_errfunc(MPI_COMM_WORLD,
			 lam_getfunc(), lam_mkerr(MPI_ERR_INTERN, 0)));

  lam_resetfunc(BLKXMPICOLOROFF);
  return MPI_SUCCESS;
}
