/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: tdelattr.c,v 1.2 2001/02/03 19:50:13 bbarrett Exp $
 *
 *	Function:	- delete attribute
 *			- detach attribute from datatype
 *	Accepts:	- datatype
 *			- attribute key
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <errno.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@
    MPI_Type_delete_attr - Delete attribue

Input Parameters:
+ type - datatype from which the attribute is deleted (handle)
- key - key value (integer)

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TYPE
.N MPI_ERR_KEYVAL
.N MPI_ERR_INTERN

.seealso MPI_Type_free_keyval(3), MPI_Type_get_attr(3), MPI_Type_set_attr(3)

.N WEB
@*/

int
MPI_Type_delete_attr(MPI_Datatype type, int key)
{
	struct _attrkey	*p;
	struct _attr	*pk;
	int		err;

	lam_initerr_m();
	lam_setfunc_m(BLKMPITDELETEATTR);
/*
 * Check the arguments.
 */
	if (type == MPI_DATATYPE_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITDELETEATTR, lam_mkerr(MPI_ERR_TYPE, 0)));
	}

	p = lam_getattr(key);
	if (p == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITDELETEATTR, lam_mkerr(MPI_ERR_KEYVAL, 0)));
	}
/*
 * Check if the key is attached to the datatype.
 */
	pk = lam_getkey(type->dt_keys, key);
	if (pk == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPITDELETEATTR, lam_mkerr(MPI_ERR_KEYVAL, 0)));
	}
/*
 * Delete the attribute via the callback function.
 */
	if (((MPI_Type_delete_attr_function *) (p->ak_del))
			!= MPI_TYPE_NULL_DELETE_FN) {

		if (p->ak_flags & LAM_LANGF77) {
			(*((MPI_F_delete_function *) (p->ak_del)))
				(&type->dt_f77handle, &key,
				(int *) &(pk->a_value),
                               	(int *) &(p->ak_extra), &err);
		} else {
			err = (*((MPI_Type_delete_attr_function *) (p->ak_del)))
				(type, key, pk->a_value, p->ak_extra);
		}

		if (err != MPI_SUCCESS) {
			return(lam_err_comm(MPI_COMM_WORLD, err, 0,
						"deleting attribute"));
		}
	}
/*
 * Detach the key from the datatype.
 */
	if (lam_delkey(type->dt_keys, key)) {
		return(lam_err_comm(MPI_COMM_WORLD, MPI_ERR_INTERN, errno,
					"deleting key"));
	}

	lam_resetfunc_m(BLKMPITDELETEATTR);
	return(MPI_SUCCESS);
}
