/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: infoset.c,v 6.7 2001/02/03 20:24:30 jsquyres Exp $
 *
 *	Function:	- set info value associated with key
 *	Accepts:	- info (inout)
 *			- key
 *			- value
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>
#include <string.h>

#include <all_list.h>
#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>


/*@
  MPI_Info_set - Set a (key, value) pair in an 'MPI_Info' object

Input Parameters:
+ key - null-terminated character string of the index key
- value - null-terminated character string of the value

Input/Output Parameter:
. info - info object (handle)

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG
.N MPI_ERR_INFO_KEY
.N MPI_ERR_INFO_NOKEY
.N MPI_ERR_INTERN

.N WEB
@*/
int MPI_Info_set(MPI_Info info, char *key, char *value)
{
	struct _infoent	newinfo;		/* new info pair */
	struct _infoent	*old;			/* existing pair with key */
	int		keylen;			/* key length */
	int		vallen;			/* value length */
	char		*newval;		/* storage for new value */

	lam_initerr_m();
	lam_setfunc_m(BLKMPIINFOSET);
/*
 * Check arguments.
 */
	if (info == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFOSET,
					lam_mkerr(MPI_ERR_ARG, 0)));
	}

	keylen = (key) ? strlen(key) : 0;
	if (keylen == 0 || keylen > MPI_MAX_INFO_KEY) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFOSET,
				lam_mkerr(MPI_ERR_INFO_KEY, 0)));
	}

	vallen = (value) ? strlen(value) : 0;
	if (vallen == 0 || vallen > MPI_MAX_INFO_VAL) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFOSET,
					lam_mkerr(MPI_ERR_INFO_VALUE, 0)));
	}
/*
 * Allocate storage for the new value.
 */
	if ((newval = (char *) malloc(vallen + 1)) == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIINFOSET,
					lam_mkerr(MPI_ERR_OTHER, 0)));
	}

	LAM_ZERO_ME(newinfo);
	strcpy(newinfo.ie_key, key);
	strcpy(newval, value);
/*
 * Check if key already appears in the info object.
 */
	old = al_find(info->info_list, &newinfo);

	if (old) {
/*
 * The key is already in the info object.  Replace the old value with the new.
 */
		free(old->ie_value);
		old->ie_value = newval;
	} else {
/*
 * Add the key/value pair to the info object.
 */
		newinfo.ie_value = newval;
		if (al_append(info->info_list, &newinfo) == 0) {
			return(lam_errfunc(MPI_COMM_WORLD,
				BLKMPIINFOSET, lam_mkerr(MPI_ERR_OTHER, 0)));
		}
	}

	lam_resetfunc_m(BLKMPIINFOSET);
	return(MPI_SUCCESS);
}
