/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: graphmap.c,v 6.3 1999/12/31 21:27:27 jsquyres Exp $
 *
 *	Function:	- map processes to graph coordinates
 *	Accepts:	- old communicator
 *			- # processes
 *			- node index array
 *			- array of edges
 *			- ptr caller's new rank
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Graph_map - Maps process to graph topology information

Input Parameters:
+ comm - input communicator (handle) 
. nnodes - number of graph nodes (integer) 
. index - integer array specifying the graph structure, see
'MPI_GRAPH_CREATE'
- edges - integer array specifying the graph structure 

Output Parameter:
. pnewrank - reordered rank of the calling process; 'MPI_UNDEFINED' if
the calling process does not belong to graph (integer)
 
.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_TOPOLOGY
.N MPI_ERR_COMM
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Graph_map(MPI_Comm comm, int nnodes, int *index, 
		  int *edges, int *pnewrank)
{
	int		errcode;
	int		myrank;

	lam_initerr();
	lam_setfunc(BLKMPIGRAPHMAP);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGRAPHMAP, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPIGRAPHMAP, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if ((nnodes < 1) || (index == 0) || (edges == 0) || (pnewrank == 0)) {
		return(lam_errfunc(comm,
			BLKMPIGRAPHMAP, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Assign new rank.
 */
	errcode = MPI_Comm_rank(comm, &myrank);
	if (errcode != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIGRAPHMAP, errcode));
	}

	*pnewrank =
		((myrank < 0) || (myrank >= nnodes)) ? MPI_UNDEFINED : myrank;

	lam_resetfunc(BLKMPIGRAPHMAP);
	return(MPI_SUCCESS);
}
