/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: gfree.c,v 6.5 1999/12/31 21:27:26 jsquyres Exp $
 *
 *	Function:	- free a group
 *	Accepts:	- group (in-out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <MPISYSF.h>
#include <rpisys.h>

/*@

MPI_Group_free - Frees a group

Input Parameter
. grp - group (handle) 

Notes:

On output, 'grp' is set to 'MPI_GROUP_NULL'.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG
.N MPI_ERR_PERM_GROUP

.N ACK
@*/
int MPI_Group_free(MPI_Group *grp)
{
	struct _proc	**p;
	int		i;
	
	lam_initerr_m();
	lam_setfunc_m(BLKMPIGROUPFREE);

	if (grp == 0 || *grp == MPI_GROUP_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGROUPFREE, lam_mkerr(MPI_ERR_GROUP, 0)));
	} else {
		if ((--(*grp)->g_refcount) == 0) {
/*
 * We don't check for the empty group before since a user may
 * legitimately free a copy of it.  The problem is now that we won't
 * catch an explicit free of MPI_GROUP_EMPTY when copies of it exist.
 */
			if (*grp == MPI_GROUP_EMPTY) {
				(*grp)->g_refcount++;
				return(lam_errfunc(MPI_COMM_WORLD,
					BLKMPIGROUPFREE,
					lam_mkerr(MPI_ERR_GROUP, 0)));
			}

			for (i = (*grp)->g_nprocs, p = (*grp)->g_procs; i > 0;
			     i--, p++) {

				if ((--(*p)->p_refcount) == 0) {
					lam_procfree(*p);
				}
			}

			if ((*grp)->g_f77handle >= 0) {
				lam_F_free_hdl((*grp)->g_f77handle);
			}

			free((char *) *grp);
		}

		*grp = MPI_GROUP_NULL;
	}

	lam_resetfunc_m(BLKMPIGROUPFREE);
	return(MPI_SUCCESS);
}
