/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: gcmp.c,v 6.5 1999/12/31 23:16:42 jsquyres Exp $
 *
 *	Function:	- compare two groups
 *	Accepts:	- first group
 *			- second group
 *			- ptr comparison result
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <stdlib.h>
#include <string.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <terror.h>

/*
 * local functions
 */
static int		proccmp();

/*@

MPI_Group_compare - Compares two groups

Input Parameters:
+ g1 - group1 (handle) 
- g2 - group2 (handle) 

Output Parameter:
. pres - integer which is 'MPI_IDENT' if the order and members of the
two groups are the same, 'MPI_SIMILAR' if only the members are the
same, and 'MPI_UNEQUAL' otherwise

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_GROUP
.N MPI_ERR_ARG
.N MPI_ERR_EXHAUSTED

.N ACK
@*/
int MPI_Group_compare(MPI_Group g1, MPI_Group g2, 
		      int *pres)
{
	int		i;			/* favourite index */
	struct _proc	**p1, **p2;		/* favourite pointers */
	struct _proc	**sort1;		/* group1 sort array */
	struct _proc	**sort2;		/* group2 sort array */

	lam_initerr();
	lam_setfunc(BLKMPIGROUPCMP);
/*
 * Check the arguments.
 */
	if ((g1 == MPI_GROUP_NULL) || (g2 == MPI_GROUP_NULL)) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGROUPCMP, lam_mkerr(MPI_ERR_GROUP, 0)));
	}

	if (pres == 0) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPIGROUPCMP, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Handle the trivial cases.
 */
	if (g1 == g2) {
		*pres = MPI_IDENT;
		lam_resetfunc(BLKMPIGROUPCMP);
		return(MPI_SUCCESS);
	}

	if (g1->g_nprocs != g2->g_nprocs) {
		*pres = MPI_UNEQUAL;
		lam_resetfunc(BLKMPIGROUPCMP);
		return(MPI_SUCCESS);
	}
/*
 * Loop checking if same processes in same order.
 */
	p1 = g1->g_procs;
	p2 = g2->g_procs;

	for (i = 0; i < g1->g_nprocs; ++i, ++p1, ++p2) {
		if (*p1 != *p2) break;
	}

	if (i == g1->g_nprocs) {
		*pres = MPI_IDENT;
		lam_resetfunc(BLKMPIGROUPCMP);
		return(MPI_SUCCESS);
	}
/*
 * Loop checking if same processes in different order.
 * Sort the processes and compare the sorted lists for equality.
 */
	i = g1->g_nprocs * sizeof(struct _proc *);
	sort1 = (struct _proc **) malloc((unsigned) (i + i));

	if (sort1 == 0) {
		return(lam_errfunc(MPI_COMM_WORLD, BLKMPIGROUPCMP,
				lam_mkerr(MPI_ERR_OTHER, errno)));
	}

	sort2 = sort1 + g1->g_nprocs;

	memcpy((char *) sort1, (char *) g1->g_procs, i);
	memcpy((char *) sort2, (char *) g2->g_procs, i);

	qsort((char *) sort1, g1->g_nprocs, sizeof(struct _proc *), proccmp);
	qsort((char *) sort2, g2->g_nprocs, sizeof(struct _proc *), proccmp);

	p1 = sort1;
	p2 = sort2;

	for (i = 0; i < g1->g_nprocs; ++i, ++p1, ++p2) {
		if (*p1 != *p2) break;
	}

	*pres = (i == g1->g_nprocs) ? MPI_SIMILAR : MPI_UNEQUAL;

	free((char *) sort1);

	lam_resetfunc(BLKMPIGROUPCMP);
	return(MPI_SUCCESS);
}

/*
 *	proccmp
 *
 *	Function:	- compare 2 process entries
 *	Accepts:	- ptr 2 process pointers
 *	Returns:	- -ve/0/+ve
 */
static int
proccmp(p1, p2)

struct _proc		**p1, **p2;

{
	return(lam_proccmp(*p1, *p2));
}
