/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1996 The Ohio State University
 *	RBD
 *
 *	$Id: cartmap.c,v 6.3 1999/12/31 21:27:18 jsquyres Exp $
 *
 *	Function:	- map processes to cartesian coordinates
 *	Accepts:	- communicator
 *			- # dimensions
 *			- # procs per dimension
 *			- dimension periodicity
 *			- caller's new rank (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>

/*@

MPI_Cart_map - Maps process to Cartesian topology information 

Input Parameters:
+ comm - input communicator (handle) 
. ndims - number of dimensions of cartesian structure (integer) 
. dims - integer array of size 'ndims' specifying the number of
processes in each coordinate direction
- periods - logical array of size 'ndims' specifying the periodicity
specification in each coordinate direction

Output Parameter:
. newrank - reordered rank of the calling process; 'MPI_UNDEFINED' if
calling process does not belong to grid (integer)

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_COMM
.N MPI_ERR_DIMS
.N MPI_ERR_ARG

.N ACK
@*/
int MPI_Cart_map(MPI_Comm comm, int ndims, int *dims, 
		 int *periods, int *newrank)
{
	int		nprocs;
	int		rank;
	int		size;
	int		errcode;
	int		i;
	int		*p;

	lam_initerr();
	lam_setfunc(BLKMPICARTMAP);
/*
 * Check the arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_errfunc(MPI_COMM_WORLD,
			BLKMPICARTMAP, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (LAM_IS_INTER(comm)) {
		return(lam_errfunc(comm,
			BLKMPICARTMAP, lam_mkerr(MPI_ERR_COMM, 0)));
	}

	if (ndims < 1) {
		return(lam_errfunc(comm,
			BLKMPICARTMAP, lam_mkerr(MPI_ERR_DIMS, 0)));
	}

	if ((dims == 0) || (periods == 0) || (newrank == 0)) {
		return(lam_errfunc(comm,
			BLKMPICARTMAP, lam_mkerr(MPI_ERR_ARG, 0)));
	}
/*
 * Compute the # of processes in the grid.
 */
	nprocs = 1;
	for (i = 0, p = dims; i < ndims; ++i, ++p) {
		if (*p <= 0) {
			return(lam_errfunc(comm, BLKMPICARTMAP,
					lam_mkerr(MPI_ERR_DIMS, 0)));
		}

		nprocs *= *p;
	}
/*
 * Check that number of processes <= size of communicator.
 */
	errcode = MPI_Comm_size(comm, &size);
	if (errcode != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPICARTMAP, errcode));
	}

	if (nprocs > size) {
		return(lam_errfunc(comm, BLKMPICARTMAP,
					lam_mkerr(MPI_ERR_DIMS, 0)));
	}
/*
 * Compute my new rank.
 */
	errcode = MPI_Comm_rank(comm, &rank);
	if (errcode != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPICARTMAP, errcode));
	}

	*newrank = ((rank < 0) || (rank >= nprocs)) ? MPI_UNDEFINED : rank;

	lam_resetfunc(BLKMPICARTMAP);
	return(MPI_SUCCESS);
}
