/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD
 *
 *	$Id: vctable.c,v 6.3 1999/05/24 16:56:52 kmeyer1 Exp $
 *
 *	Function:	- manipulate the table of virtual circuits
 *
 *			- _vcinsert(): insert a new virtual circuit
 *			- _vcfind(): locate a virtual circuit entry
 *			- _vcdelete(): delete a virtual circuit entry
 */

#include <portable.h>
#include <net.h>
#include <terror.h>
#include <typical.h>
#include <vctable.h>

#include <stdlib.h>
#include <unistd.h>

/*
 * external variables
 */
extern int		errno;

/*
 * local variables
 */
static int		f_init = 1;		/* initialization flag */
static struct vcdesc	vctable[VCMAX];		/* VC hash table */

/*
 *	_vcinsert
 *
 *	Function:	- insert a virtual circuit in the table
 *			- initialize the table if necessary
 *			- does not allow duplicate entries
 *	Accepts:	- node ID
 *			- event
 *			- type
 *			- VC path
 *	Returns:	- 0 or ERROR
 */
int
_vcinsert(node, event, type, path)

int4			node;
int4			event;
int4			type;
int4			path;

{
	int4		i, j;		/* favourite indices */
	struct vcdesc	*p;		/* favourite pointer */
/*
 * Initialize the table if first time function is called.
 */
	if (f_init) {
		f_init = 0;
		for (i = 0, p = vctable; i < VCMAX; ++i, ++p) {
			p->vc_node = NOTNODEID;
		}
	}

	if (node == NOTNODEID) {
		errno = EBADNODE;
		return(ERROR);
	}
/*
 * Locate an empty slot in the hash table.
 * Duplicate entries are not allowed.
 */
	i = _vchash(node, event, type);
	p = vctable + i;

	for (j = 0; j < VCMAX; ++j) {
		if (p->vc_node == NOTNODEID) {
/*
 * Found empty slot.
 */
			break;
		}
		else if ((p->vc_node == node) &&
			(p->vc_event == event) && (p->vc_type == type)) {
/*
 * Found duplicate entry.
 */
			errno = EINVAL;
			return(ERROR);
		}

		if (++i == VCMAX) {
			i = 0;
			p = vctable;
		} else {
			++p;
		}
	}
/*
 * If table is full, return an error.
 */
	if (j == VCMAX) {
		errno = EFULL;
		return(ERROR);
	}
/*
 * Fill the empty slot.
 */
	p->vc_node = node;
	p->vc_event = event;
	p->vc_type = type;
	p->vc_path = path;

	return(0);
}

/*
 *	_vcfind
 *
 *	Function:	- locate a virtual circuit in the table
 *	Accepts:	- node ID
 *			- event
 *			- type
 *	Returns:	- ptr to VC entry or NULL
 */
struct vcdesc *
_vcfind(node, event, type)

int4			node;
int4			event;
int4			type;

{
	int4		i, j;		/* favourite indices */
	struct vcdesc	*p;		/* favourite pointer */

	if (node == NOTNODEID) {
		errno = EBADNODE;
		return((struct vcdesc *) 0);
	}
/*
 * Locate the entry in the hash table.
 */
	i = _vchash(node, event, type);
	p = vctable + i;

	for (j = 0; j < VCMAX; ++j) {
		if ((p->vc_node == node) &&
			(p->vc_event == event) && (p->vc_type == type)) {
/*
 * The entry is found.
 */
			return(p);
		}

		if (++i == VCMAX) {
			i = 0;
			p = vctable;
		} else {
			++p;
		}
	}
/*
 * The entry does not exist.
 */
	errno = EINVAL;
	return((struct vcdesc *) 0);
}

/*
 *	_vcdelete
 *
 *	Function:	- delete a virtual circuit from the table
 *	Accepts:	- node ID
 *			- event
 *			- type
 *	Returns:	- 0 or ERROR
 */
int
_vcdelete(node, event, type)

int4			node;
int4			event;
int4			type;

{
	struct vcdesc	*p;		/* favourite pointer */
/*
 * Locate the correct VC entry in the table
 */
	p = _vcfind(node, event, type);
	if (p == 0) {
		return(ERROR);
	}
/*
 * Erase the entry.
 */
	p->vc_node = NOTNODEID;

	return(0);
}

/*
 *	_vcfree
 *
 *	Function:	- free all virtual circuits
 */
void
_vcfree()

{
	int4		i;		/* favourite index */
	struct vcdesc	*p;		/* favourite pointer */
/*
 * Free all V.C. entries.
 */
	f_init = 1;

	for (i = 0, p = vctable; i < VCMAX; ++i, ++p) {
		p->vc_node = NOTNODEID;
		close(p->vc_path);
	}

}
