/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1994 The Ohio State University
 *	RBD
 *
 *	$Id: bootsockio.c,v 6.3 2000/03/06 21:13:23 jsquyres Exp $
 *
 *	Function:	- OTB booting socket I/O functions
 *				- readsockint4(): read an int4 from a socket
 *				- writesockint4(): read an int4 to a socket
 *				- readcltcoord(): read client nodeid/port #
 *				- readcltnbr(): read client neighbour info
 *				- writecltcoord(): write client nodeid/port #
 *				- writecltnbr(): write client neighbour info
 */

#include <errno.h>
#include <portable.h>
#include <t_types.h>
#include <typical.h>

/*
 * external functions
 */
extern int		mread();
extern int		mwrite();

/*
 *	readsockint4
 *
 *	Function:	- read a 32-bit integer from a socket
 *			- handles byte ordering issue
 *	Accepts:	- socket
 *			- ptr to integer
 *	Returns:	- 0 or ERROR
 */
int
readsockint4(sock, pdata)

int			sock;
int4			*pdata;

{
	int4		data;
	int		ret;

	ret = mread(sock, (char *) &data, sizeof(int4));
	if (ret < sizeof(int4)) {
		errno = (ret < 0) ? errno : EIO;
		return(ERROR);
	}

	*pdata = ttol(data);
	return(0);
}

/*
 *	writesockint4
 *
 *	Function:	- write a 32-bit integer to a socket
 *			- handles byte ordering issue
 *	Accepts:	- socket
 *			- integer
 *	Returns:	- 0 or ERROR
 */
int
writesockint4(sock, data)

int			sock;
int4			data;

{
	int		ret;

	data = ltot(data);
	ret = mwrite(sock, (char *) &data, sizeof(int4));
	if (ret < sizeof(int4)) {
		errno = (ret < 0) ? errno : EIO;
		return(ERROR);
	}

	return(0);
}

/*
 *	readcltcoord
 *
 *	Function:	- read client STREAM and UDP port numbers
 *	Accepts:	- socket
 *			- ptr to STREAM port #
 *			- ptr to UDP port #
 *	Returns:	- 0 or ERROR
 */
int
readcltcoord(sock, pstrport, pudpport)

int			sock;
int4			*pstrport;
int4			*pudpport;

{
	int4		status;

	if (readsockint4(sock, &status)) {
		return(ERROR);
	}

	if (status) {
		errno = status;
		return(ERROR);
	}

	if (readsockint4(sock, pstrport)) {
		return(ERROR);
	}

	if (readsockint4(sock, pudpport)) {
		return(ERROR);
	}

	return(0);
}

/*
 *	writecltcoord
 *
 *	Function:	- write client STREAM and UDP port numbers
 *			- called by the booted client
 *	Accepts:	- socket
 *			- STREAM port #
 *			- UDP port #
 *	Returns:	- 0 or ERROR
 */
int
writecltcoord(sock, strport, udpport)

int			sock;
int4			strport;
int4			udpport;

{
	if (writesockint4(sock, INT4_NIL)) {
		return(ERROR);
	}

	if (writesockint4(sock, strport)) {
		return(ERROR);
	}

	if (writesockint4(sock, udpport)) {
		return(ERROR);
	}

	return(0);
}

/*
 *	readcltnbr
 *
 *	Function:	- read client neighbour <link index, hostaddr, port,
 *                                               ncpus>
 *	Accepts:	- socket
 *			- ptr link index
 *			- ptr host address (4 bytes, in network byte order)
 *			- ptr port number
 *			- ptr num cpus
 *	Returns:	- 0 or ERROR
 */
int
readcltnbr(sock, plink, phostaddr, pport, pnum_cpus)

int			sock;
int4			*plink;
unsigned char		*phostaddr;
int4			*pport;
int4			*pnum_cpus;

{
	int		ret;

	if (readsockint4(sock, plink)) {
		return(ERROR);
	}

	ret = mread(sock, (char *) phostaddr, 4);
	if (ret < 4) {
		errno = (ret < 0) ? errno : EIO;
		return(ERROR);
	}

	if (readsockint4(sock, pport)) {
		return(ERROR);
	}

	if (readsockint4(sock, pnum_cpus)) {
		return(ERROR);
	}

	return(0);
}

/*
 *	writecltnbr
 *
 *	Function:	- write client neighbour <link index, hostaddr, port, 
 *                                                ncpus>
 *			- called by the booting server
 *	Accepts:	- socket
 *			- link index
 *			- host address (4 bytes, in network byte order)
 *			- port number
 *			- num cpus
 *	Returns:	- 0 or ERROR
 */
int
writecltnbr(sock, link, hostaddr, port, num_cpus)

int			sock;
int4			link;
unsigned char		*hostaddr;
int4			port;
int4			num_cpus;

{
	int		ret;

	if (writesockint4(sock, link)) {
		return(ERROR);
	}

	ret = mwrite(sock, (char *) hostaddr, 4);
	if (ret < 4) {
		errno = (ret < 0) ? errno : EIO;
		return(ERROR);
	}

	if (writesockint4(sock, port)) {
		return(ERROR);
	}

	if (writesockint4(sock, num_cpus)) {
		return(ERROR);
	}

	return(0);
}
