/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *
 *	$Id: rpi.shm.h.in,v 6.4 1999/08/09 12:50:37 lamteam Exp $
 *
 *	Function:	- shared memory / TCP client-to-client interface header
 */

#ifndef _RPI_SHM_H
#define _RPI_SHM_H

#include <errno.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/sem.h>

#include <mpi.h>
#include <rpi.c2c.h>

/*
 * constants
 */
#define CACHELINESIZE		64		/* cache line size (or more) */
#define INITHASH		((int4) 59)	/* initial hash table size */
#define TCPSHORTMSGLEN		65536	/* max. length tcp short msg */

#define LAM_MPI_SHMPOOLSIZE	0	/* size of long message pool */
#define	LAM_MPI_SHMMAXALLOC	0	/* max allocation of shmem */
#define SHMSHORTMSGLEN		8192	/* max. length shm short msg */
#define SHMBOXSIZE		(SHMSHORTMSGLEN + CACHELINESIZE)

/*
 * constants for shared memory allocator internals
 */
#define LOG_ALIGN	4
#define ALIGNMENT	(1 << LOG_ALIGN)

/*
 * public variables
 */
extern char		*_shm_membase;

/*
 * polling and locking
 */
extern int		_shm_poll_yield;

#if LAM_RPI_SHM_SYSV

/*
 * shared memory postbox
 */
struct boxhdr {
	struct c2c_envl	bh_env;
	int		bh_size;
	int		bh_bufoff;
};

union postbox {
	struct boxhdr	pb_header;
	char		pb_pad[CACHELINESIZE];
};

typedef union postbox	postbox_t;

/*
 * client-to-client specific process data
 */
struct c2c_proc {
	struct c2c_envl	cp_env;			/* incoming envelope */
	int		cp_write;		/* ok to try to write flag */
	int		cp_read;		/* ok to try to read flag */
	int		cp_sock;		/* socket descriptor */
	char		*cp_envbuf;		/* envelope buffer pointer */
	int		cp_nenvin;		/* # env. bytes left to read */
	char		*cp_msgbuf;		/* message buffer pointer */
	int		cp_nmsgin;		/* # msg. bytes left to read */
	int		cp_extra;		/* # bytes to go into sink */
	int		cp_shm;			/* shm block id */
	int		cp_sem;			/* semaphore set id */
	int		cp_ppsem;		/* process pair sema set id */
	int		cp_locked;		/* reader has a lock? */
	int		cp_shmidx;		/* shared memory index */
	int		cp_insize;		/* incoming area size */
	postbox_t	*cp_inbox;		/* incoming postbox */
	postbox_t	*cp_outbox;		/* outgoing postbox */
	struct sembuf	*cp_lop;		/* lock operations */
	struct sembuf	*cp_top;		/* try lock operations */
	struct sembuf	*cp_uop;		/* unlock operations */
	MPI_Request	cp_wreq;		/* req. writing to process */
	MPI_Request	cp_rreq;		/* reading req. */
	MPI_Request	cp_mreq;		/* req. to test matching from */
	int		(*cp_readfn)();		/* read function */
	struct cbuf_msg *cp_bmsg;		/* buffering? */
	int		cp_nbfde;		/* # buffered envelopes */
};

/*
 * locking macros
 */
#define _shm_readtrylock(p)  \
	((semop((p)->cp_ppsem, (p)->cp_top, 1)) ? \
		((errno == EAGAIN) ? 1 : -1) : 0)

#define _shm_writetrylock(p) \
	((semop((p)->cp_ppsem, (p)->cp_top + 1, 1)) ? \
		((errno == EAGAIN) ? 1 : -1) : 0)

#define _shm_readunlock(p)	semop((p)->cp_ppsem, (p)->cp_uop, 1)
#define _shm_writeunlock(p)	semop((p)->cp_ppsem, (p)->cp_uop + 1, 1)

/*
 * prototypes of LAM MPI library client-to-client internal functions
 */
#ifndef __ARGS
#if __STDC__ || defined(c_plusplus) || defined(__cplusplus)
#define __ARGS(a)	a
#else
#define __ARGS(a)	()
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif

extern int	_shm_readlock __ARGS((struct c2c_proc *));
extern int	_shm_writelock __ARGS((struct c2c_proc *));

#ifdef __cplusplus
}
#endif


#elif LAM_RPI_SHM_USYSV

/*
 * polling and locking
 */
extern int		_lock_poll_read;
extern int		_lock_poll_write;
extern int		_shm_poll_yield;

/*
 * shared memory postbox
 */
struct boxhdr {
	struct c2c_envl	bh_env;
	int		bh_size;
	int		bh_bufoff;
	volatile int	bh_lock;
};

union postbox {
	struct boxhdr	pb_header;
	char		pb_pad[CACHELINESIZE];
};

typedef union postbox	postbox_t;

/*
 * client-to-client specific process data
 */
struct c2c_proc {
	struct c2c_envl	cp_env;			/* incoming envelope */
	int		cp_write;		/* ok to try to write flag */
	int		cp_read;		/* ok to try to read flag */
	int		cp_sock;		/* socket descriptor */
	char		*cp_envbuf;		/* envelope buffer pointer */
	int		cp_nenvin;		/* # env. bytes left to read */
	char		*cp_msgbuf;		/* message buffer pointer */
	int		cp_nmsgin;		/* # msg. bytes left to read */
	int		cp_extra;		/* # bytes to go into sink */
	int		cp_shm;			/* id of shm block */
	int		cp_sem;			/* id of semaphore set */
	int		cp_shmidx;		/* shared memory index */
	postbox_t	*cp_inbox;		/* incoming postbox */
	postbox_t	*cp_outbox;		/* outgoing postbox */
	int		cp_insize;		/* incoming area size */
	MPI_Request	cp_wreq;		/* req. writing to process */
	MPI_Request	cp_rreq;		/* reading req. */
	MPI_Request	cp_mreq;		/* req. to test matching from */
	int		(*cp_readfn)();		/* read function */
	struct cbuf_msg *cp_bmsg;		/* buffering? */
	int		cp_nbfde;		/* # buffered envelopes */
};

#else

Error: Invalid RPI type.

#endif

/*
 * prototypes of LAM MPI library client-to-client internal functions
 */
#ifdef __cplusplus
extern "C" {
#endif

int	_shm_create_area(int, struct c2c_proc *, struct nmsg *);
int	_shm_attach_area(int, struct c2c_proc *, struct nmsg *);
int	_shm_advance(void);
int	_shm_req_rcvd_ack_long(struct c2c_proc *, MPI_Request);
int	_shm_req_send_ack_long(struct c2c_proc *, MPI_Request);
int	_shm_req_send_ack_only(struct c2c_proc *, MPI_Request);
int	_shm_req_send_body_first(struct c2c_proc *, MPI_Request);
int	_shm_req_done_synch(struct c2c_proc *, MPI_Request);
int	_shm_proc_read_body_box(struct c2c_proc *);
int	_shm_proc_read_body_pool(struct c2c_proc *);

/*
 * public functions
 */
int	_shm_buffer(struct c2c_proc *);
int	_shm_match_adv(struct c2c_proc *);
int	_shm_proc_read_env(struct c2c_proc *);
int	_shm_req_send_long(struct c2c_proc *, MPI_Request);
int	_shm_req_send_short(struct c2c_proc *, MPI_Request);
int	_shm_req_send_synch(struct c2c_proc *, MPI_Request);
int	_shm_req_rcvd_body_synch(struct c2c_proc *, MPI_Request);
int	_shmtcp_req_probe(struct c2c_proc *, MPI_Request);
int	_shmtcp_req_recv(struct c2c_proc *, MPI_Request);
int	_shm_push_env(struct c2c_proc *, MPI_Request);
int	_shm_push_body_pool(struct c2c_proc *, MPI_Request);
int	_shm_push_body_box(struct c2c_proc *, MPI_Request);
int	_shm_req_rcvd_long_ack(struct c2c_proc *, MPI_Request);  
int	_shm_buffered_adv(MPI_Request, struct cbuf_msg *);
int	_shm_fastsend(char *, int, struct c2c_proc *, int, int, MPI_Comm);
int	_shm_fastrecv(char *, int *, struct c2c_proc *, int, int *, 
			   MPI_Comm, int *);

int	lam_shmalloc(unsigned int *, int *);
void	lam_shfree(void *);

#ifdef __cplusplus
}
#endif

#endif	/* _RPI_SHM_H */
