//////////////////////////////////////////////////////////////////////////
//                                                                      //
// XrdClient                                                            //
//                                                                      //
// Author: Fabrizio Furano (INFN Padova, 2004)                          //
// Adapted from TXNetFile (root.cern.ch) originally done by             //
//  Alvise Dorigo, Fabrizio Furano                                      //
//          INFN Padova, 2003                                           //
//                                                                      //
// A UNIX reference client for xrootd.                                  //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

//         $Id: XrdClient.cc,v 1.69 2006/05/23 13:28:46 ganis Exp $

#include "XrdClient/XrdClient.hh"
#include "XrdClient/XrdClientDebug.hh"
#include "XrdClient/XrdClientUrlSet.hh"
#include "XrdClient/XrdClientConn.hh"
#include "XrdClient/XrdClientEnv.hh"
#include "XrdClient/XrdClientConnMgr.hh"
#include "XrdClient/XrdClientSid.hh"

#include <stdio.h>
#ifndef WIN32
#include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>

XrdOucSemWait     XrdClient::fConcOpenSem(DFLT_MAXCONCURRENTOPENS);
kXR_unt16         XrdClient::fgNmax = 0;
int               XrdClient::fgSrvProtocol = -1;

//_____________________________________________________________________________
// Calls the Open func in order to parallelize the Open requests
//
void *FileOpenerThread(void *arg, XrdClientThread *thr) {
   // Function executed in the garbage collector thread
   XrdClient *thisObj = (XrdClient *)arg;

   thr->SetCancelDeferred();
   thr->SetCancelOn();

   thisObj->TryOpen(thisObj->fOpenPars.mode, thisObj->fOpenPars.options, false);

   return 0;
}


//_____________________________________________________________________________
XrdClient::XrdClient(const char *url) {
    fReadAheadLast = 0;
    fOpenerTh = 0;
    fOpenProgCnd = new XrdOucCondVar(0);
    fReadWaitData = new XrdOucCondVar(0);

    memset(&fStatInfo, 0, sizeof(fStatInfo));
    memset(&fOpenPars, 0, sizeof(fOpenPars));

   // Pick-up the latest setting of the debug level
   DebugSetLevel(EnvGetLong(NAME_DEBUG));

   int CacheSize = EnvGetLong(NAME_READCACHESIZE);

    fUseCache = (CacheSize > 0);

    if (!ConnectionManager)
	Info(XrdClientDebug::kNODEBUG,
	     "Create",
	     "(C) 2004 SLAC INFN XrdClient " << XRD_CLIENT_VERSION);

#ifndef WIN32
    signal(SIGPIPE, SIG_IGN);
#endif

    fInitialUrl = url;

    fConnModule = new XrdClientConn();


    if (!fConnModule) {
	Error("Create","Object creation failed.");
	abort();
    }

    fConnModule->SetRedirHandler(this);
}

//_____________________________________________________________________________
XrdClient::~XrdClient()
{
    // Terminate the opener thread

    fOpenProgCnd->Lock();

    if (fOpenerTh) {
	delete fOpenerTh;
	fOpenerTh = 0;
    }

    fOpenProgCnd->UnLock();


    Close();

    if (fConnModule)
	delete fConnModule;

    delete fReadWaitData;
    delete fOpenProgCnd;
}

//_____________________________________________________________________________
bool XrdClient::IsOpen_inprogress()
{
   // Non blocking access to the 'inprogress' flag
   bool res;

   if (!fOpenProgCnd) return false;

   fOpenProgCnd->Lock();
   res = fOpenPars.inprogress;
   fOpenProgCnd->UnLock();

   return res;
};

//_____________________________________________________________________________
bool XrdClient::IsOpen_wait() {
    bool res;

    if (!fOpenProgCnd) return false;

    fOpenProgCnd->Lock();

    if (fOpenPars.inprogress) {
	fOpenProgCnd->Wait();
	if (fOpenerTh) {
	    delete fOpenerTh;
	    fOpenerTh = 0;
	}
    }
    res = fOpenPars.opened;
    fOpenProgCnd->UnLock();

    return res;
};

//_____________________________________________________________________________
void XrdClient::TerminateOpenAttempt() {
    fOpenProgCnd->Lock();

    fOpenPars.inprogress = false;
    fOpenProgCnd->Broadcast();
    fOpenProgCnd->UnLock();

    fConcOpenSem.Post();

    //cout << "Mytest " << time(0) << " File: " << fUrl.File << " - Open finished." << endl;
}

//_____________________________________________________________________________
bool XrdClient::Open(kXR_unt16 mode, kXR_unt16 options, bool doitparallel) {
    short locallogid;
  
    // But we initialize the internal params...
    fOpenPars.opened = FALSE;  
    fOpenPars.options = options;
    fOpenPars.mode = mode;  

    // Now we try to set up the first connection
    // We cycle through the list of urls given in fInitialUrl
  

    // Max number of tries
    int connectMaxTry = EnvGetLong(NAME_FIRSTCONNECTMAXCNT);
  
    // Construction of the url set coming from the resolution of the hosts given
    XrdClientUrlSet urlArray(fInitialUrl);
    if (!urlArray.IsValid()) {
	Error("Open", "The URL provided is incorrect.");
	return FALSE;
    }

    //
    // Now start the connection phase, picking randomly from UrlArray
    //
    urlArray.Rewind();
    locallogid = -1;
    int urlstried = 0;
    for (int connectTry = 0;
	 (connectTry < connectMaxTry) && (!fConnModule->IsConnected()); 
	 connectTry++) {

	XrdClientUrlInfo *thisUrl = 0;
	urlstried = (urlstried == urlArray.Size()) ? 0 : urlstried;

	bool nogoodurl = TRUE;
	while (urlArray.Size() > 0) {

	    // Get an url from the available set
	    if ((thisUrl = urlArray.GetARandomUrl())) {

		if (fConnModule->CheckHostDomain(thisUrl->Host,
						 EnvGetString(NAME_CONNECTDOMAINALLOW_RE),
						 EnvGetString(NAME_CONNECTDOMAINDENY_RE))) {
		    nogoodurl = FALSE;

		    Info(XrdClientDebug::kHIDEBUG, "Open", "Trying to connect to " <<
			 thisUrl->Host << ":" << thisUrl->Port << ". Connect try " <<
			 connectTry+1);
		    locallogid = fConnModule->Connect(*thisUrl, this);
		    // To find out if we have tried the whole URLs set
		    urlstried++;
		    break;
		} else {
		    // Invalid domain: drop the url and move to next, if any
		    urlArray.EraseUrl(thisUrl);
		    continue;
		}
	    }
	}
	if (nogoodurl) {
	    Error("Open", "Access denied to all URL domains requested");
	    break;
	}

	// We are connected to a host. Let's handshake with it.
	if (fConnModule->IsConnected()) {

	    // Now the have the logical Connection ID, that we can use as streamid for 
	    // communications with the server

	    Info(XrdClientDebug::kHIDEBUG, "Open",
		 "The logical connection id is " << fConnModule->GetLogConnID() <<
		 ".");

	    fConnModule->SetUrl(*thisUrl);
	    fUrl = *thisUrl;
        
	    Info(XrdClientDebug::kHIDEBUG, "Open", "Working url is " << thisUrl->GetUrl());
        
	    // after connection deal with server
	    if (!fConnModule->GetAccessToSrv())
           
		if (fConnModule->LastServerError.errnum == kXR_NotAuthorized) {
		    if (urlstried == urlArray.Size()) {
			// Authentication error: we tried all the indicated URLs:
			// does not make much sense to retry
			fConnModule->Disconnect(TRUE);
			XrdOucString msg(fConnModule->LastServerError.errmsg);
			msg.erasefromend(1);
			Error("Open", "Authentication failure: " << msg);
			break;
		    } else {
			XrdOucString msg(fConnModule->LastServerError.errmsg);
			msg.erasefromend(1);
			Info(XrdClientDebug::kHIDEBUG, "Open",
			     "Authentication failure: " << msg);
		    }
		} else {
		    Error("Open", "Access to server failed: error: " <<
			  fConnModule->LastServerError.errnum << " (" << 
			  fConnModule->LastServerError.errmsg << ") - retrying.");
		}
	    else {
		Info(XrdClientDebug::kUSERDEBUG, "Open", "Access to server granted.");
		break;
	    }
	}
     
	// The server denied access. We have to disconnect.
	Info(XrdClientDebug::kHIDEBUG, "Open", "Disconnecting.");
     
	fConnModule->Disconnect(FALSE);
     
	if (connectTry < connectMaxTry-1) {

	    if (DebugLevel() >= XrdClientDebug::kUSERDEBUG)
		Info(XrdClientDebug::kUSERDEBUG, "Open",
		     "Connection attempt failed. Sleeping " <<
		     EnvGetLong(NAME_RECONNECTTIMEOUT) << " seconds.");
     
	    sleep(EnvGetLong(NAME_RECONNECTTIMEOUT));

	}

    } //for connect try


    if (!fConnModule->IsConnected()) {
	return FALSE;
    }

  
    //
    // Variable initialization
    // If the server is a new xrootd ( load balancer or data server)
    //
    if ((fConnModule->GetServerType() != XrdClientConn::kSTRootd) && 
	(fConnModule->GetServerType() != XrdClientConn::kSTNone)) {
	// Now we are connected to a server that didn't redirect us after the 
	// login/auth phase
	// let's continue with the openfile sequence

	Info(XrdClientDebug::kUSERDEBUG,
	     "Open", "Opening the remote file " << fUrl.File); 

	if (!TryOpen(mode, options, doitparallel)) {
	    Error("Open", "Error opening the file " <<
		  fUrl.File << " on host " << fUrl.Host << ":" <<
		  fUrl.Port);

	    return FALSE;

	} else {

	    if (doitparallel) {
		Info(XrdClientDebug::kUSERDEBUG, "Open", "File open in progress.");
	    }
	    else
		Info(XrdClientDebug::kUSERDEBUG, "Open", "File opened succesfully.");

	}

    } else {
	// the server is an old rootd
	if (fConnModule->GetServerType() == XrdClientConn::kSTRootd) {
	    return FALSE;
	}
	if (fConnModule->GetServerType() == XrdClientConn::kSTNone) {
	    return FALSE;
	}
    }

    return TRUE;

}

//_____________________________________________________________________________
int XrdClient::Read(void *buf, long long offset, int len) {
    XrdClientIntvList cacheholes;
    long blkstowait;

    Info( XrdClientDebug::kHIDEBUG, "Read",
	  "Read(offs=" << offset <<
	  ", len=" << len << ")" );

    if (!IsOpen_wait()) {
	Error("Read", "File not opened.");
	return 0;
    }

    if (!fUseCache) {
	// Without caching

	// Prepare a request header 
	ClientRequest readFileRequest;
	memset( &readFileRequest, 0, sizeof(readFileRequest) );
	fConnModule->SetSID(readFileRequest.header.streamid);
	readFileRequest.read.requestid = kXR_read;
	memcpy( readFileRequest.read.fhandle, fHandle, sizeof(fHandle) );
	readFileRequest.read.offset = offset;
	readFileRequest.read.rlen = len;
	readFileRequest.read.dlen = 0;

	fConnModule->SendGenCommand(&readFileRequest, 0, 0, (void *)buf,
				    FALSE, (char *)"ReadBuffer");

	return fConnModule->LastServerResp.dlen;
    }


    // Ok, from now on we are sure that we have to deal with the cache
    struct XrdClientStatInfo stinfo;
    Stat(&stinfo);
    len = xrdmax(0, xrdmin(len, stinfo.size - offset));

    kXR_int32 rasize = EnvGetLong(NAME_READAHEADSIZE);

    bool retrysync = false;

    // we cycle until we get all the needed data
    do {

	cacheholes.Clear();
	blkstowait = 0;
	long bytesgot = 0;

	if (!retrysync) {


	    bytesgot = fConnModule->GetDataFromCache(buf, offset,
						     len + offset - 1,
						     true,
						     cacheholes, blkstowait);

	    Info(XrdClientDebug::kHIDEBUG, "Read",
		 "Cache response: got " << bytesgot << " bytes. Holes= " <<
		 cacheholes.GetSize() << " Outstanding= " << blkstowait);

	    // If the cache gives the data to us
	    //  we don't need to ask the server for them... in principle!
	    if( bytesgot >= len ) {

		// The cache gave us all the requested data

		Info(XrdClientDebug::kHIDEBUG, "Read",
		     "Found data in cache. len=" << len <<
		     " offset=" << offset);

		// Are we using read ahead?
		if ( EnvGetLong(NAME_GOASYNC) &&
		     // We read ahead only if the last byte we got is near (or over) to the last byte read
		     // in advance. But not too much over.
		     (fReadAheadLast - (offset+len) < rasize) &&
                     (fReadAheadLast - (offset+len) > -10*rasize) &&
		     (rasize > 0) ) {

		    kXR_int64 araoffset;
		    kXR_int32 aralen;

		    // This is a HIT case. Async readahead will try to put some data
		    // in advance into the cache. The higher the araoffset will be,
		    // the best chances we have not to cause overhead
		    araoffset = xrdmax(fReadAheadLast, offset + len);
		    aralen = xrdmin(rasize,
				    offset + len + rasize -
				    xrdmax(offset + len, fReadAheadLast));

		    if (aralen > 0) {
			TrimReadRequest(araoffset, aralen, rasize);
			fReadAheadLast = araoffset + aralen;
			Read_Async(araoffset, aralen);
		    }
		}

		return len;
	    }



	

	    // We are here if the cache did not give all the data to us
	    // We should have a list of blocks to request
	    for (int i = 0; i < cacheholes.GetSize(); i++) {
		kXR_int64 offs;
		kXR_int32 len;
	    
		offs = cacheholes[i].beginoffs;
		len = cacheholes[i].endoffs - offs + 1;


		Info( XrdClientDebug::kHIDEBUG, "Read",
		      "Hole in the cache: offs=" << offs <<
		      ", len=" << len );
	    
		Read_Async(offs, len);
	    }
	
	
	    // Are we using read ahead?
	    if ( EnvGetLong(NAME_GOASYNC) &&
		 // We read ahead only if the last byte we got is near (or over) to the last byte read
		 // in advance. But not too much over.
		 (fReadAheadLast - (offset+len) < rasize) &&
                 (fReadAheadLast - (offset+len) > -10*rasize) &&
		 (rasize > 0) ) {

		kXR_int64 araoffset;
		kXR_int32 aralen;

		// This is a HIT case. Async readahead will try to put some data
		// in advance into the cache. The higher the araoffset will be,
		// the best chances we have not to cause overhead
		araoffset = xrdmax(fReadAheadLast, offset + len);
		aralen = xrdmin(rasize,
				offset + len + rasize -
				xrdmax(offset + len, fReadAheadLast));

		if (aralen > 0) {
		    TrimReadRequest(araoffset, aralen, rasize);
		    fReadAheadLast = araoffset + aralen;
		    Read_Async(araoffset, aralen);
		}
	    }

	}

	// If we got nothing from the cache let's do it sync and exit!
	// Note that this part has the side effect of triggering the recovery actions
	//  if we get here after an error (or timeout)
	// Hence it's not a good idea to make async also this read
	// Remember also that a sync read request must not be modified if it's going to be
	//  written into the application-given buffer
	if (retrysync || (!bytesgot && !blkstowait && !cacheholes.GetSize())) {

	    retrysync = false;

	    Info( XrdClientDebug::kHIDEBUG, "Read",
		  "Read(offs=" << offset <<
		  ", len=" << len << "). Going sync." );

	    // Prepare a request header 
	    ClientRequest readFileRequest;
	    memset( &readFileRequest, 0, sizeof(readFileRequest) );
	    fConnModule->SetSID(readFileRequest.header.streamid);
	    readFileRequest.read.requestid = kXR_read;
	    memcpy( readFileRequest.read.fhandle, fHandle, sizeof(fHandle) );
	    readFileRequest.read.offset = offset;
	    readFileRequest.read.rlen = len;
	    readFileRequest.read.dlen = 0;

	    fConnModule->SendGenCommand(&readFileRequest, 0, 0, (void *)buf,
					FALSE, (char *)"ReadBuffer");

	    return fConnModule->LastServerResp.dlen;
	}

	// Now it's time to sleep
	// This thread will be awakened when new data will arrive
	if ((blkstowait > 0)|| cacheholes.GetSize()) {
	    Info( XrdClientDebug::kHIDEBUG, "Read",
		  "Waiting " << blkstowait+cacheholes.GetSize() << "outstanding blocks." );

	    fReadWaitData->Lock();

	    if (fReadWaitData->Wait(10)) {

		Info( XrdClientDebug::kUSERDEBUG, "Read",
		  "Timeout waiting outstanding blocks. Retrying sync!" );
		
		retrysync = true;
	    }

	    fReadWaitData->UnLock();

	}
	
	

    } while ((blkstowait > 0) || cacheholes.GetSize());

    return len;
}

//_____________________________________________________________________________
kXR_int64 XrdClient::ReadV(char *buf, kXR_int64 *offsets, int *lens, int nbuf)
{
    // This is only an additional step to ReadVEach... it will only split
    // a really big request in multiple request... it sounds kind of silly
    // since we are trying to do the opposite but there are restrictions
    // in the server that dont let us send such big buffers.
    // Note: this could return a smaller buffer than expected (for example
    // if only a few readings were allowed).

    // We say that the range for an unsigned int of 16 bits is 0 to +65,535
    // which is Request.read.dlen .
    // Since we are only sending "readahead_list" which are 16 bytes we could 
    // send 4095 requests in one single call...

    // Maximum number of the kXR_unt16 type
    if (fgNmax == 0) {
       kXR_int32 limit = 1 << (int)(sizeof(kXR_unt16) * 8);
       fgNmax  = (kXR_unt16) ((limit) / sizeof(struct readahead_list));
    }
    int i = 0;
    kXR_int64 pos = 0; 
    kXR_int64 res = 0;
    int n = ( fgNmax < nbuf  ) ? fgNmax : nbuf;

    while ( i < nbuf ) {
       if ( (res = ReadVEach(&buf[pos], &offsets[i], &lens[i], n)) <= 0)
          break;
       pos += res;
       i += n;

       if ( (nbuf - i) < n  )
          n = (nbuf - i);
    }

    // pos will indicate the size of the data read
    // Even if we were able to read only a part of the buffer !!!
    return pos;
}

//_____________________________________________________________________________
kXR_int64 XrdClient::ReadVEach(char *buf, kXR_int64 *offsets, int *lens, int nbuf) {
    // This will compress multiple request in one single read to
    // avoid the latency.
    // We dont know how to deal with the cache in this case of reading
    // so we leave that for a near future (the same for read-aheads)...
    // this call is completely sync. and straightforward

    Info( XrdClientDebug::kHIDEBUG, "ReadV", " Number of Buffers=" << nbuf);

    for(int i = 0 ; i < nbuf ; i++){
      Info( XrdClientDebug::kHIDEBUG, "ReadV",
            "Read(offs=" << offsets[i] << ", len=" << lens[i] << ")" ); 
    }

    if (!IsOpen_wait()) {
       Error("ReadV", "File not opened.");
       return 0;
    }

    // The first thing to do is to check if the server version
    // supports the vectored reads... has to be > 2.4.5
    if (fgSrvProtocol < 0) {
       // The first time retrieve it
       ClientRequest protocolFileRequest;
       memset( &protocolFileRequest, 0, sizeof(protocolFileRequest) );
       fConnModule->SetSID(protocolFileRequest.header.streamid);
       protocolFileRequest.protocol.requestid = kXR_protocol; // is it ok?
       protocolFileRequest.header.requestid = kXR_protocol;  
       protocolFileRequest.protocol.dlen = 0;

       ServerResponseBody_Protocol responseBody;

       fConnModule->SendGenCommand(&protocolFileRequest, 0, 0, 
                                  (void *)&responseBody, FALSE, (char *)"Protocol");

       fgSrvProtocol = ntohl(responseBody.pval);
       responseBody.flags  = ntohl(responseBody.flags);
    }

    // This means problems in getting the protocol version
    if ( fgSrvProtocol < 0 ) {
       Info(XrdClientDebug::kHIDEBUG, "ReadV",
            "Problems retrieving protocol version run by the server" );
       return -1;
    }

    // This means the server won't support it
    if ( fgSrvProtocol < 0x00000246 ) {
       Info(XrdClientDebug::kHIDEBUG, "ReadV",
            "The server is an old version and doesn't support vectored reading" );
       return -1;
    }

    kXR_int64 res;
    kXR_int64 pos_from = 0;
    kXR_int64 pos_to = 0;

    char *buflis = new char[nbuf * sizeof(struct readahead_list)];

    // Here we put the information of all the buffers in a single list
    // then it's up to server to interpret it and send us all the data
    // in a single buffer
    int ofs = 0;
    kXR_int64 total_len = 0;
    for (int i = 0; i < nbuf; i++) {
        struct readahead_list rhl;
        memcpy( &(rhl.fhandle), fHandle, sizeof(fHandle) ); 
        rhl.offset = offsets[i];
        rhl.rlen = lens[i];
        total_len += lens[i];
        // Save into the buffer
        memcpy(buflis + ofs, &rhl, sizeof(struct readahead_list)); 
        ofs += sizeof(struct readahead_list);
    }

    // Prepare a request header 
    ClientRequest readFileRequest;
    memset( &readFileRequest, 0, sizeof(readFileRequest) );
    fConnModule->SetSID(readFileRequest.header.streamid);
    readFileRequest.read.requestid = kXR_read;
    memcpy( readFileRequest.read.fhandle, fHandle, sizeof(fHandle) );
    readFileRequest.read.offset = 0;
    readFileRequest.read.rlen = 0;
    readFileRequest.read.dlen = nbuf * sizeof(struct readahead_list);
    readFileRequest.read.flags = kXR_DoReadV;
    readFileRequest.read.path = 0;

    char *res_buf = new char[total_len + (nbuf * sizeof(struct readahead_list))];

    fConnModule->SendGenCommand(&readFileRequest, buflis, 0, 
                               (void *)res_buf, FALSE, (char *)"ReadBuffer");

    delete [] buflis;

    // This probably means that the server doesnt support ReadV
    // ( old version of the server  )
    if( fConnModule->LastServerResp.status == kXR_error){
       res = -1;
       goto end;
    }
    res = fConnModule->LastServerResp.dlen;

    // I dont do anything with the info we get from the server
    // I just rebuild the readahead_list element
    struct readahead_list header;
    pos_from = 0;
    pos_to = 0;
    for ( int i = 0; i < nbuf ; i++) {
       memcpy(&header, res_buf + pos_from, sizeof(struct readahead_list));
       
       kXR_int64 tmpl;
       memcpy(&tmpl, &header.offset, sizeof(kXR_int64) );
       tmpl = ntohll(tmpl);
       memcpy(&header.offset, &tmpl, sizeof(kXR_int64) );

       header.rlen  = ntohl(header.rlen);       
       
       // If the data we receive is not the data we asked for we might
       // be seeing an error... but it has to be handled in a different
       // way if we get the data in a different order.
       if ( offsets[i] != header.offset || lens[i] != header.rlen ) {
	  res = -1;
	  goto end;
       }

       pos_from += sizeof(struct readahead_list);
       memcpy( &buf[pos_to], &res_buf[pos_from], header.rlen);
       pos_from += header.rlen;
       pos_to += header.rlen;
    }
    res = pos_to;
    
 end:
    delete [] res_buf;
    return res;
}

//_____________________________________________________________________________
bool XrdClient::Write(const void *buf, long long offset, int len) {

    if (!IsOpen_wait()) {
	Error("WriteBuffer", "File not opened.");
	return FALSE;
    }


    // Prepare request
    ClientRequest writeFileRequest;
    memset( &writeFileRequest, 0, sizeof(writeFileRequest) );
    fConnModule->SetSID(writeFileRequest.header.streamid);
    writeFileRequest.write.requestid = kXR_write;
    memcpy( writeFileRequest.write.fhandle, fHandle, sizeof(fHandle) );
    writeFileRequest.write.offset = offset;
    writeFileRequest.write.dlen = len;
   
   
    return fConnModule->SendGenCommand(&writeFileRequest, buf, 0, 0,
				       FALSE, (char *)"Write");
}


//_____________________________________________________________________________
bool XrdClient::Sync()
{
    // Flushes un-written data

 
    if (!IsOpen_wait()) {
	Error("Sync", "File not opened.");
	return FALSE;
    }


    // Prepare request
    ClientRequest flushFileRequest;
    memset( &flushFileRequest, 0, sizeof(flushFileRequest) );

    fConnModule->SetSID(flushFileRequest.header.streamid);

    flushFileRequest.sync.requestid = kXR_sync;

    memcpy(flushFileRequest.sync.fhandle, fHandle, sizeof(fHandle));

    flushFileRequest.sync.dlen = 0;

    return fConnModule->SendGenCommand(&flushFileRequest, 0, 0, 0, 
                                       FALSE, (char *)"Sync");
  
}

//_____________________________________________________________________________
bool XrdClient::TryOpen(kXR_unt16 mode, kXR_unt16 options, bool doitparallel) {
   
    int thrst = 0;

    fOpenPars.inprogress = true;

    if (doitparallel) {

	for (int i = 0; i < DFLT_MAXCONCURRENTOPENS; i++) {

	    fConcOpenSem.Wait();
	    fOpenerTh = new XrdClientThread(FileOpenerThread);

	    thrst = fOpenerTh->Run(this);     
	    if (!thrst) {
		// The thread start seems OK. This open will go in parallel

		if (fOpenerTh->Detach())
		    Error("XrdClient", "Thread detach failed. Low system resources?");

		return true;
	    }

	    // Note: the Post() here is intentionally missing.

	    Error("XrdClient", "Parallel open thread start failed. Low system"
		  " resources? Res=" << thrst << " Count=" << i);
	    delete fOpenerTh;
	    fOpenerTh = 0;

	}

	// If we are here it seems that this machine cannot start open threads at all
	// In this desperate situation we try to go sync anyway.
	for (int i = 0; i < DFLT_MAXCONCURRENTOPENS; i++) fConcOpenSem.Post();

	Error("XrdClient", "All the parallel open thread start attempts failed."
	      " Desperate situation. Going sync.");
     
	doitparallel = false;
    }

    // First attempt to open a remote file
    bool lowopenRes = LowOpen(fUrl.File.c_str(), mode, options);
    if (lowopenRes) {
	TerminateOpenAttempt();
	return TRUE;
    }

    // If the open request failed for the error "file not found" proceed, 
    // otherwise return FALSE
    if (fConnModule->GetOpenError() != kXR_NotFound) {
	TerminateOpenAttempt();
	return FALSE;
    }


    // If connected to a host saying "File not Found" or similar then...

    // If we are currently connected to a host which is different
    // from the one we formerly connected, then we resend the request
    // specifyng the supposed failing server as opaque info
    if (fConnModule->GetLBSUrl() &&
	(fConnModule->GetCurrentUrl().Host != fConnModule->GetLBSUrl()->Host) ) {
	XrdOucString opinfo;

	opinfo = "&tried=" + fConnModule->GetCurrentUrl().Host;

	Info(XrdClientDebug::kUSERDEBUG,
	     "Open", "Back to " << fConnModule->GetLBSUrl()->Host <<
	     ". Refreshing cache. Opaque info: " << opinfo);

	if ( (fConnModule->GoToAnotherServer(*fConnModule->GetLBSUrl()) == kOK) &&
	     LowOpen(fUrl.File.c_str(), mode, options | kXR_refresh,
		     (char *)opinfo.c_str() ) ) {
	    TerminateOpenAttempt();
	    return TRUE;
	}
	else {

	    Error("Open", "Error opening the file.");
	    TerminateOpenAttempt();
	    return FALSE;
	}

    }

    TerminateOpenAttempt();
    return FALSE;

}

//_____________________________________________________________________________
bool XrdClient::LowOpen(const char *file, kXR_unt16 mode, kXR_unt16 options,
			char *additionalquery) {

    // Low level Open method
    XrdOucString finalfilename(file);

    if (fConnModule->fRedirOpaque.length() > 0) {
	finalfilename += "?";
	finalfilename += fConnModule->fRedirOpaque;
    }

    if (additionalquery)
	finalfilename += additionalquery;

    // Send a kXR_open request in order to open the remote file
    ClientRequest openFileRequest;

    struct ServerResponseBody_Open openresp;

    memset(&openFileRequest, 0, sizeof(openFileRequest));

    fConnModule->SetSID(openFileRequest.header.streamid);

    openFileRequest.header.requestid = kXR_open;

    // Now set the options field basing on user's requests
    openFileRequest.open.options = options;

    // Set the open mode field
    openFileRequest.open.mode = mode;

    // Set the length of the data (in this case data describes the path and 
    // file name)
    openFileRequest.open.dlen = finalfilename.length();

    // Send request to server and receive response
    bool resp = fConnModule->SendGenCommand(&openFileRequest,
					    (const void *)finalfilename.c_str(),
					    0, &openresp, FALSE, (char *)"Open");

    if (resp) {
	// Get the file handle to use for future read/write...
	memcpy( fHandle, openresp.fhandle, sizeof(fHandle) );

	fOpenPars.opened = TRUE;
	fOpenPars.options = options;
	fOpenPars.mode = mode;
    
    }

    return fOpenPars.opened;
}

//_____________________________________________________________________________
bool XrdClient::Stat(struct XrdClientStatInfo *stinfo) {

    if (!IsOpen_wait()) {
	Error("Stat", "File not opened.");
	return FALSE;
    }

    if (fStatInfo.stated) {
	if (stinfo)
	    memcpy(stinfo, &fStatInfo, sizeof(fStatInfo));
	return TRUE;
    }
   
    // asks the server for stat file informations
    ClientRequest statFileRequest;
   
    memset(&statFileRequest, 0, sizeof(ClientRequest));
   
    fConnModule->SetSID(statFileRequest.header.streamid);
   
    statFileRequest.stat.requestid = kXR_stat;
    memset(statFileRequest.stat.reserved, 0, 
	   sizeof(statFileRequest.stat.reserved));

    statFileRequest.stat.dlen = fUrl.File.length();
   
    char fStats[2048];
    memset(fStats, 0, 2048);

    bool ok = fConnModule->SendGenCommand(&statFileRequest,
					  (const char*)fUrl.File.c_str(),
					  0, fStats , FALSE, (char *)"Stat");
   






    if (ok && (fConnModule->LastServerResp.status == 0) ) {

	Info(XrdClientDebug::kHIDEBUG,
	     "Stat", "Returned stats=" << fStats);
   
	sscanf(fStats, "%ld %lld %ld %ld",
	       &fStatInfo.id,
	       &fStatInfo.size,
	       &fStatInfo.flags,
	       &fStatInfo.modtime);

	if (stinfo)
	    memcpy(stinfo, &fStatInfo, sizeof(fStatInfo));

	fStatInfo.stated = true;
    }

    return ok;
}

//_____________________________________________________________________________
bool XrdClient::Close() {

    if (!IsOpen_wait()) {
	Info(XrdClientDebug::kUSERDEBUG, "Close", "File not opened.");
	return TRUE;
    }

    ClientRequest closeFileRequest;
  
    memset(&closeFileRequest, 0, sizeof(closeFileRequest) );

    fConnModule->SetSID(closeFileRequest.header.streamid);

    closeFileRequest.close.requestid = kXR_close;
    memcpy(closeFileRequest.close.fhandle, fHandle, sizeof(fHandle) );
    closeFileRequest.close.dlen = 0;
  
    fConnModule->SendGenCommand(&closeFileRequest, 0,
				0, 0, FALSE, (char *)"Close");
  
    // No file is opened for now
    fOpenPars.opened = FALSE;

    return TRUE;
}


//_____________________________________________________________________________
bool XrdClient::OpenFileWhenRedirected(char *newfhandle, bool &wasopen)
{
    // Called by the comm module when it needs to reopen a file
    // after a redir

    wasopen = fOpenPars.opened;

    if (!fOpenPars.opened)
	return TRUE;

    fOpenPars.opened = FALSE;

    Info(XrdClientDebug::kHIDEBUG,
	 "OpenFileWhenRedirected", "Trying to reopen the same file." );

    kXR_unt16 options = fOpenPars.options;

    if (fOpenPars.options & kXR_delete) {
	Info(XrdClientDebug::kHIDEBUG,
	     "OpenFileWhenRedirected", "Stripping off the 'delete' option." );

	options &= !kXR_delete;
	options |= kXR_open_updt;
    }

    if (fOpenPars.options & kXR_new) {
	Info(XrdClientDebug::kHIDEBUG,
	     "OpenFileWhenRedirected", "Stripping off the 'new' option." );

	options &= !kXR_new;
	options |= kXR_open_updt;
    }

    if ( TryOpen(fOpenPars.mode, options, false) ) {

	fOpenPars.opened = TRUE;

	Info(XrdClientDebug::kHIDEBUG,
	     "OpenFileWhenRedirected",
	     "Open successful." );

	memcpy(newfhandle, fHandle, sizeof(fHandle));

	return TRUE;
    } else {
	Error("OpenFileWhenRedirected", 
	      "New redir destination server refuses to open the file.");
      
	return FALSE;
    }
}

//_____________________________________________________________________________
bool XrdClient::Copy(const char *localpath) {

    if (!IsOpen_wait()) {
	Error("Copy", "File not opened.");
	return FALSE;
    }

    Stat(0);
    int f = open(localpath, O_CREAT | O_RDWR);   
    if (f < 0) {
	Error("Copy", "Error opening local file.");
	return FALSE;
    }

    void *buf = malloc(100000);
    long long offs = 0;
    int nr = 1;

    while ((nr > 0) && (offs < fStatInfo.size))
	if ( (nr = Read(buf, offs, 100000)) )
	    offs += write(f, buf, nr);
	 
    close(f);
    free(buf);
   
    return TRUE;
}

//_____________________________________________________________________________
UnsolRespProcResult XrdClient::ProcessUnsolicitedMsg(XrdClientUnsolMsgSender *sender,
						     XrdClientMessage *unsolmsg) {
    // We are here if an unsolicited response comes from a logical conn
    // The response comes in the form of an TXMessage *, that must NOT be
    // destroyed after processing. It is destroyed by the first sender.
    // Remember that we are in a separate thread, since unsolicited 
    // responses are asynchronous by nature.

    if (unsolmsg->GetStatusCode() != XrdClientMessage::kXrdMSC_ok) {
	Info(XrdClientDebug::kHIDEBUG,
	     "ProcessUnsolicitedMsg", "Incoming unsolicited communication error message." );
    }
    else {
	Info(XrdClientDebug::kHIDEBUG,
	     "ProcessUnsolicitedMsg", "Incoming unsolicited response from streamid " <<
	     unsolmsg->HeaderSID() );
    }

    // Local processing ....

    if (unsolmsg->IsAttn()) {
	struct ServerResponseBody_Attn *attnbody;

	attnbody = (struct ServerResponseBody_Attn *)unsolmsg->GetData();

        int actnum = (attnbody) ? (attnbody->actnum) : 0;

	// "True" async resp is processed here
	switch (actnum) {

	case kXR_asyncdi:
	    // Disconnection + delayed reconnection request

	    struct ServerResponseBody_Attn_asyncdi *di;
	    di = (struct ServerResponseBody_Attn_asyncdi *)unsolmsg->GetData();

	    // Explicit redirection request
	    if (di) {
		Info(XrdClientDebug::kUSERDEBUG,
		     "ProcessUnsolicitedMsg", "Requested Disconnection + Reconnect in " <<
		     ntohl(di->wsec) << " seconds.");

		fConnModule->SetRequestedDestHost((char *)fUrl.Host.c_str(), fUrl.Port);
		fConnModule->SetREQDelayedConnectState(ntohl(di->wsec));
	    }

	    // Other objects may be interested in this async resp
	    return kUNSOL_CONTINUE;
	    break;
	 
	case kXR_asyncrd:
	    // Redirection request

	    struct ServerResponseBody_Attn_asyncrd *rd;
	    rd = (struct ServerResponseBody_Attn_asyncrd *)unsolmsg->GetData();

	    // Explicit redirection request
	    if (rd && (strlen(rd->host) > 0)) {
		Info(XrdClientDebug::kUSERDEBUG,
		     "ProcessUnsolicitedMsg", "Requested redir to " << rd->host <<
		     ":" << ntohl(rd->port));

		fConnModule->SetRequestedDestHost(rd->host, ntohl(rd->port));
	    }

	    // Other objects may be interested in this async resp
	    return kUNSOL_CONTINUE;
	    break;

	case kXR_asyncwt:
	    // Puts the client in wait state

	    struct ServerResponseBody_Attn_asyncwt *wt;
	    wt = (struct ServerResponseBody_Attn_asyncwt *)unsolmsg->GetData();

	    if (wt) {
		Info(XrdClientDebug::kUSERDEBUG,
		     "ProcessUnsolicitedMsg", "Pausing client for " << ntohl(wt->wsec) <<
		     " seconds.");

		fConnModule->SetREQPauseState(ntohl(wt->wsec));
	    }

	    // Other objects may be interested in this async resp
	    return kUNSOL_CONTINUE;
	    break;

	case kXR_asyncgo:
	    // Resumes from pause state

	    Info(XrdClientDebug::kUSERDEBUG,
		 "ProcessUnsolicitedMsg", "Resuming from pause.");

	    fConnModule->SetREQPauseState(0);

	    // Other objects may be interested in this async resp
	    return kUNSOL_CONTINUE;
	    break;

	case kXR_asynresp:
	    // A response to a request which got a kXR_waitresp as a response
	
	    // We pass it direcly to the connmodule for processing
	    // The processing will tell if the streamid matched or not,
	    // in order to stop further processing
	    return fConnModule->ProcessAsynResp(unsolmsg);
	    break;

	default:

	    Info(XrdClientDebug::kUSERDEBUG,
	         "ProcessUnsolicitedMsg", "Empty message");

	    // Propagate the message
	    return kUNSOL_CONTINUE;

	} // switch

      
    }
    else
	// Let's see if the message is a communication error message
	if (unsolmsg->GetStatusCode() != XrdClientMessage::kXrdMSC_ok)
	    return fConnModule->ProcessAsynResp(unsolmsg);
	else
	    // Let's see if we are receiving the response to an async read request
	    if ( SidManager->JoinedSids(fConnModule->GetStreamID(), unsolmsg->HeaderSID()) ) {
		struct SidInfo *si = SidManager->GetSidInfo(unsolmsg->HeaderSID());
		ClientRequest *req = &(si->outstandingreq);
	 
		Info(XrdClientDebug::kHIDEBUG,
		     "ProcessUnsolicitedMsg",
		     "Processing unsolicited response from streamid " <<
		     unsolmsg->HeaderSID() << " father=" <<
		     si->fathersid );
	 
		if ( (req->header.requestid == kXR_read) &&
		     ( (unsolmsg->HeaderStatus() == kXR_oksofar) || 
		       (unsolmsg->HeaderStatus() == kXR_ok) ) ) {
	    
		    long long offs = req->read.offset + si->reqbyteprogress;
	    
		    Info(XrdClientDebug::kHIDEBUG, "ProcessUnsolicitedMsg",
			 "Putting data into cache. Offset=" <<
			 offs <<
			 " len " <<
			 unsolmsg->fHdr.dlen);

		    fReadWaitData->Lock();

		    // To compute the end offset of the block we have to take 1 from the size!
		    fConnModule->SubmitDataToCache(unsolmsg, offs,
						   offs + unsolmsg->fHdr.dlen - 1);

		    fReadWaitData->Broadcast();
		    fReadWaitData->UnLock();

		    si->reqbyteprogress += unsolmsg->fHdr.dlen;
	    
		    if (unsolmsg->HeaderStatus() == kXR_ok) return kUNSOL_DISPOSE;
		    else return kUNSOL_KEEP;
		}
	 
	    }
   
   
    return kUNSOL_CONTINUE;
}

XReqErrorType XrdClient::Read_Async(long long offset, int len) {

    if (!IsOpen_wait()) {
	Error("Read", "File not opened.");
	return kGENERICERR;
    }

    if (!len) return kOK;

    if (fUseCache)
	fConnModule->SubmitPlaceholderToCache(offset, offset+len-1);

    // Prepare request
    ClientRequest readFileRequest;
    memset( &readFileRequest, 0, sizeof(readFileRequest) );

    // No need to initialize the streamid, it will be filled by XrdClientConn
    readFileRequest.read.requestid = kXR_read;
    memcpy( readFileRequest.read.fhandle, fHandle, sizeof(fHandle) );
    readFileRequest.read.offset = offset;
    readFileRequest.read.rlen = len;
    readFileRequest.read.dlen = 0;



    Info(XrdClientDebug::kHIDEBUG, "Read_Async",
	 "Requesting to read " <<
	 readFileRequest.read.rlen <<
	 " bytes of data at offset " <<
	 readFileRequest.read.offset);

     

    return (fConnModule->WriteToServer_Async(&readFileRequest, 0));


}


bool XrdClient::TrimReadRequest(kXR_int64 &offs, kXR_int32 &len, kXR_int32 rasize) {

    kXR_int64 newoffs;
    kXR_int32 newlen, minlen, blksz;

    if (!fUseCache ) return false;

    blksz = xrdmax(rasize, 16384);

    newoffs = offs / blksz * blksz;

    minlen = (offs + len - newoffs);
    newlen = ((minlen / blksz + 1) * blksz);


    newlen = xrdmax(rasize, newlen);

    if (fConnModule->CacheWillFit(newlen)) {
	offs = newoffs;
	len = newlen;
	return true;
    }

    return false;

}

//_____________________________________________________________________________
// Sleeps on a condvar which is signalled when a new async block arrives
void XrdClient::WaitForNewAsyncData() {

   fReadWaitData->Lock();
   fReadWaitData->Wait();
   fReadWaitData->UnLock();

}

//_____________________________________________________________________________
bool XrdClient::UseCache(bool u)
{
  // Set use of cache flag after checking if the requested value make sense.
  // Returns the previous value to allow quick toggling of the flag.

  bool r = fUseCache;

  if (!u) {
    fUseCache = FALSE;
  } else {
    if (EnvGetLong(NAME_READCACHESIZE) > 0)
      fUseCache = TRUE;
  }
  // Return the previous setting
  return r;
}
