// @(#)root/base:$Name: v5-12-00e $:$Id: TMacro.cxx,v 1.9 2006/07/03 16:10:43 brun Exp $
// Author: Rene Brun   16/08/2005

/*************************************************************************
 * Copyright (C) 1995-2005, Rene Brun and Fons Rademakers.               *
 * All rights reserved.                                                  *
 *                                                                       *
 * For the licensing terms see $ROOTSYS/LICENSE.                         *
 * For the list of contributors see $ROOTSYS/README/CREDITS.             *
 *************************************************************************/


//////////////////////////////////////////////////////////////////////////
//                                                                      //
// TMacro                                                               //
//                                                                      //
// Class supporting a collection of lines with C++ code.                //
// A TMacro can be executed, saved to a ROOT file, edited, etc.         //
//                                                                      //
// A macro can be built line by line by calling the AddLine function.   //
// or it can be created directly from a file via the special constructor//
// when the first argument is a file name.                              //
//                                                                      //
// A macro can be executed via the Exec function.                       //
// Arguments can be specified when calling Exec.                        //
//                                                                      //
// A macro can be drawn in a pad. When the pad is updated, the macro is //
// automatically executed.                                              //
//                                                                      //
// The code in the macro can be saved via the SaveSource function.      //
// If the macro is in the list of primitives of a pad/canvas, the macro //
// will be saved in the script generated by TCanvas::SaveSource.        //
//                                                                      //
// A macro can be written to a ROOT file via TObject::Write.            //
//                                                                      //
// Examples:                                                            //
//   TMacro m("Peaks.C");  //macro m with name "Peaks" is created       //
//                         //from file  Peaks.C                         //
//   m.Exec();             //macro executed with default arguments      //
//   m.Exec("4");          //macro executed with argument               //
//   m.SaveSource("newPeaks.C");                                        //
//   TFile f("mymacros.root","recreate");                               //
//   m.Write();   //macro saved to file with name "Peaks"               //
//                                                                      //
//////////////////////////////////////////////////////////////////////////

#include "Riostream.h"
#include "TROOT.h"
#include "TEnv.h"
#include "TSystem.h"
#include "TMacro.h"
#include "TMD5.h"

ClassImp(TMacro)

//______________________________________________________________________________
TMacro::TMacro(): TNamed()
{
   // Create an empty macro, use AddLine() or ReadFile() to fill this macro.

   fLines  = 0;
   fParams = "";
}

//______________________________________________________________________________
TMacro::TMacro(const char *name, const char *title)
       :TNamed(name,title)
{
   // Create a macro with a name and a title.
   // If name is the name of a file, the macro is automatically filled
   // by reading all the lines in the file. In this case, if the title
   // is empty, it will be the name of the file.

   fLines  = new TList();
   fParams = "";
   if (!name) return;
   Int_t nch = strlen(name);
   char *s = new char[nch+1];
   strcpy(s,name);
   char *dot   = (char*)strchr(s,'.');
   char *slash = (char*)strrchr(s,'/');
   if (dot) {
      *dot = 0;
      if (slash) fName = slash+1;
      else       fName = s;
      if (fTitle.Length() == 0) fTitle = name;
      ReadFile(name);
   }
   delete [] s;
}

//______________________________________________________________________________
TMacro::TMacro(const TMacro &macro): TNamed(macro)
{
   // Copy constructor.

   fLines = new TList();
   TIter next(macro.GetListOfLines());
   TObjString *obj;
   while ((obj = (TObjString*) next())) {
      fLines->Add(new TObjString(obj->GetName()));
   }
}

//______________________________________________________________________________
TMacro::~TMacro()
{
   // Delete this macro.

   if (fLines) fLines->Delete();
   delete fLines;
}
 
//______________________________________________________________________________
TMacro& TMacro::operator=(const TMacro &macro)
{
   // Copy constructor.
  
   if(this!=&macro) {
      TNamed::operator=(macro);
      if(fLines) delete fLines;
      fLines = new TList();
      TIter next(macro.GetListOfLines());
      TObjString *obj;
      while ((obj = (TObjString*) next())) {
         fLines->Add(new TObjString(obj->GetName()));
      }
   }
   return *this;
}

//______________________________________________________________________________
TObjString *TMacro::AddLine(const char *text)
{
   // Add line with text in the list of lines of this macro.

   if (!fLines) fLines = new TList();
   TObjString *obj = new TObjString(text);
   fLines->Add(obj);
   return obj;
}

//______________________________________________________________________________
void TMacro::Browse(TBrowser * /*b*/)
{
   // When clicking in the browser, the following action is performed
   // on this macro, depending the content of the variable TMacro.Browse.
   // TMacro.Browse can be set in the system.rootrc or .rootrc file like
   //     TMacro.Browse   :  Action
   // or set via gEnv->SetValue, eg
   //     gEnv->SetValue("TMacro.Browse","Print");
   // By default TMacro.Browse=""
   // -if TMacro.Browse ="" the macro is executed
   // -if TMacro.Browse ="Print" the macro is printed in stdout
   // -if TMacro.Browse is of the form "mymacro.C"
   //     the macro void mymacro.C(TMacro *m) is called where m=this macro
   //     An example of macro.C saving the macro into a file and viewing it
   //     with emacs is shown below:
   //        void mymacro(TMacro *m) {
   //           m->SaveSource("xx.log");
   //           gSystem->Exec("emacs xx.log&");
   //        }

   TString opt = gEnv->GetValue("TMacro.Browse","");
   if (opt.IsNull()) {
      Exec();
      return;
   }
   if (opt == "Print") {
      Print();
      return;
   }
   if (opt.Contains(".C")) {
      const char *cmd = Form(".x %s((TMacro*)0x%x)",opt.Data(),this);
      gROOT->ProcessLine(cmd);
      return;
   }
}

//______________________________________________________________________________
TMD5 *TMacro::Checksum()
{
   // Returns checksum of the current content. The returned TMD5 object must
   // be deleted by the user. Returns 0 in case of error.

   if (!fLines || fLines->GetSize() <= 0)
      return (TMD5 *)0;

   TMD5 *md5 = new TMD5;

   // Fill (same params as in TMD5::FileChecksum)
   const Int_t bufSize = 8192;
   UChar_t buf[bufSize];
   Long64_t pos = 0;
   Long64_t left = bufSize;

   TIter nxl(fLines);
   TObjString *l;
   while ((l = (TObjString *) nxl())) {
      TString line = l->GetString();
      line += '\n';
      Int_t len = line.Length();
      char *p = (char *) line.Data();
      if (left > len) {
         strncpy((char *)&buf[pos], p, len);
         pos += len;
         left -= len;
      } else if (left == len) {
         strncpy((char *)&buf[pos], p, len);
         md5->Update(buf, bufSize);
         pos = 0;
         left = bufSize;
      } else {
         strncpy((char *)&buf[pos], p, left);
         md5->Update(buf, bufSize);
         len -= left;
         p += left;
         pos = 0;
         left = bufSize;
         strncpy((char *)&buf[pos], p, len);
         pos += len;
         left -= len;
      }
   }
   md5->Update(buf, pos);

   // Finalize
   md5->Final();

   return md5;
}

//______________________________________________________________________________
void TMacro::Exec(const char *params)
{
   // Execute this macro with params, if params is 0, default parameters
   // (set via SetParams) are used.

   //the current implementation uses a file in the current directory.
   //should be replaced by a direct execution from memory by CINT
   TString fname = GetName();
   fname += ".Cexec";
   SaveSource(fname);
   //disable a possible call to gROOT->Reset from the executed script
   gROOT->SetExecutingMacro(kTRUE);
   //execute script in /tmp
   TString exec = ".x " + fname;
   TString p = params;
   if (p == "") p = fParams;
   if (p != "")
      exec += "(" + p + ")";
   gROOT->ProcessLine(exec);
   //enable gROOT->Reset
   gROOT->SetExecutingMacro(kFALSE);
   //delete the temporary file
   gSystem->Unlink(fname);
}

//______________________________________________________________________________
TObjString *TMacro::GetLineWith(const char *text) const
{
   // Search the first line containing text.

   if (!fLines) return 0;
   TIter next(fLines);
   TObjString *obj;
   while ((obj = (TObjString*) next())) {
      if (strstr(obj->GetName(),text)) return obj;
   }
   return 0;
}

//______________________________________________________________________________
void TMacro::Paint(Option_t *option)
{
   // Execute this macro (called by TPad::Paint).

   Exec(option);
}

//______________________________________________________________________________
void TMacro::Print(Option_t * /*option*/) const
{
   // Print contents of this macro.

   if (!fLines) return;
   TIter next(fLines);
   TObjString *obj;
   while ((obj = (TObjString*) next())) {
      printf("%s\n",obj->GetName());
   }
}

//______________________________________________________________________________
Int_t TMacro::ReadFile(const char *filename)
{
   // Read lines in filename in this macro.

   if (!fLines) fLines = new TList();
   ifstream in;
   in.open(filename);
   if (!in.good()) {
      Error("ReadFile","Cannot open file: %s",filename);
      return 0;
   }
   char *line = new char[10000];
   Int_t nlines = 0;
   while (1) {
      in.getline(line,10000);
      if (in.eof()) break;
      fLines->Add(new TObjString(line));
      nlines++;
   }
   delete [] line;
   return nlines;
}

//______________________________________________________________________________
void TMacro::SaveSource(const char *filename)
{
   // Save macro source in filename.

   ofstream out;
   out.open(filename, ios::out);
   if (!out.good ()) {
      Printf("SaveSource cannot open file: %s",filename);
      return;
   }
   if (!fLines) {out.close(); return;}
   TIter next(fLines);
   TObjString *obj;
   while ((obj = (TObjString*) next())) {
      out<<obj->GetName()<<endl;
   }
   out.close();
}

//______________________________________________________________________________
void TMacro::SavePrimitive(ostream &out, Option_t *option /*= ""*/)
{
   // Save macro source on stream out.

   char quote = '"';
   out<<"   "<<endl;
   if (gROOT->ClassSaved(TMacro::Class())) {
      out<<"   ";
   } else {
      out<<"   "<<ClassName()<<" *";
   }
   out<<"macro = new "<<ClassName()<<"("<<quote<<GetName()<<quote<<","<<quote<<GetTitle()<<quote<<");"<<endl;
   if (!fLines) return;
   TIter next(fLines);
   TObjString *obj;
   while ((obj = (TObjString*) next())) {
      TString s = obj->GetName();
      s.ReplaceAll("\"","\\\"");
      out<<"   macro->AddLine("<<quote<<s.Data()<<quote<<");"<<endl;
   }
   out<<"   macro->Draw("<<quote<<option<<quote<<");"<<endl;
}

//______________________________________________________________________________
void TMacro::SetParams(const char *params)
{
   // Set default parameters to execute this macro.

   if (params) fParams = params;
}
