/*****************************************************************************
 * Project: RooFit                                                           *
 * Package: RooFitCore                                                       *
 *    File: $Id: RooPlot.rdl,v 1.33 2006/07/03 15:37:11 wverkerke Exp $
 * Authors:                                                                  *
 *   WV, Wouter Verkerke, UC Santa Barbara, verkerke@slac.stanford.edu       *
 *   DK, David Kirkby,    UC Irvine,         dkirkby@uci.edu                 *
 *                                                                           *
 * Copyright (c) 2000-2005, Regents of the University of California          *
 *                          and Stanford University. All rights reserved.    *
 *                                                                           *
 * Redistribution and use in source and binary forms,                        *
 * with or without modification, are permitted according to the terms        *
 * listed in LICENSE (http://roofit.sourceforge.net/license.txt)             *
 *****************************************************************************/
#ifndef ROO_PLOT
#define ROO_PLOT

#include "TH1.h"
#include "RooList.h"
#include "RooPrintable.h"

class RooAbsReal;
class RooAbsRealLValue;
class RooArgSet ;
class RooHist;
class RooCurve ;
class RooPlotable;
class TAttLine;
class TAttFill;
class TAttMarker;
class TAttText;
class TClass ;

class RooPlot : public TH1, public RooPrintable {
public:
  RooPlot(const char* name, const char* title, const RooAbsRealLValue &var, Double_t xmin, Double_t xmax, Int_t nBins) ;
  RooPlot(const RooAbsRealLValue &var, Double_t xmin, Double_t xmax, Int_t nBins);
  RooPlot(Double_t xmin= 0, Double_t xmax= 1);
  RooPlot(Double_t xmin, Double_t xmax, Double_t ymin, Double_t ymax);
  RooPlot(const RooAbsRealLValue &var1, const RooAbsRealLValue &var2);
  RooPlot(const RooAbsRealLValue &var1, const RooAbsRealLValue &var2,
	  Double_t xmin, Double_t xmax, Double_t ymin, Double_t ymax);
  virtual ~RooPlot();

  // implement the TH1 interface
  virtual Stat_t GetBinContent(Int_t) const;
  virtual Stat_t GetBinContent(Int_t, Int_t) const;
  virtual Stat_t GetBinContent(Int_t, Int_t, Int_t) const;
  virtual void Draw(Option_t *options= 0);

  // container management
  const char* nameOf(Int_t idx) const ;
  TObject *findObject(const char *name, const TClass* clas=0) const;
  TObject* getObject(Int_t idx) const ;
  Stat_t numItems() const {return _items.GetSize();}

  void addPlotable(RooPlotable *plotable, Option_t *drawOptions= "", Bool_t invisible=kFALSE, Bool_t refreshNorm=kFALSE);
  void addObject(TObject* obj, Option_t* drawOptions= "", Bool_t invisible=kFALSE);
  void addTH1(TH1 *hist, Option_t* drawOptions= "", Bool_t invisible=kFALSE);

  // ascii printing
  virtual void printToStream(ostream& os, PrintOption opt= Standard, TString indent= "") const;
  inline virtual void Print(Option_t *options= 0) const {
    printToStream(defaultStream(),parseOptions(options));
  }

  // data member get/set methods
  inline RooAbsRealLValue *getPlotVar() const { return _plotVarClone; }
  inline Double_t getFitRangeNEvt() const { return _normNumEvts; }
  Double_t getFitRangeNEvt(Double_t xlo, Double_t xhi) const ;
  inline Double_t getFitRangeBinW() const { return _normBinWidth; }
  inline Double_t getPadFactor() const { return _padFactor; }
  inline void setPadFactor(Double_t factor) { if(factor >= 0) _padFactor= factor; }
  void updateNormVars(const RooArgSet &vars);
  const RooArgSet *getNormVars() const { return _normVars; }

  // get attributes of contained objects
  TAttLine *getAttLine(const char *name=0) const;
  TAttFill *getAttFill(const char *name=0) const;
  TAttMarker *getAttMarker(const char *name=0) const;
  TAttText *getAttText(const char *name=0) const;

  // Convenient type-safe accessors
  RooCurve* getCurve(const char* name=0) const ;
  RooHist* getHist(const char* name=0) const ;

  // rearrange drawing order of contained objects
  Bool_t drawBefore(const char *before, const char *target);
  Bool_t drawAfter(const char *after, const char *target);

  // get/set drawing options for contained objects
  TString getDrawOptions(const char *name) const;
  Bool_t setDrawOptions(const char *name, TString options);

  Bool_t getInvisible(const char* name) const ;
  void setInvisible(const char* name, Bool_t flag=kTRUE) ; 
 
  virtual void SetMaximum(Double_t maximum = -1111) ;
  virtual void SetMinimum(Double_t minimum = -1111) ;

  Double_t chiSquare(int nFitParam=0) const { return chiSquare(0,0,nFitParam) ; } 
  Double_t chiSquare(const char* pdfname, const char* histname, int nFitParam=0) const ;

  RooHist* residHist(const char* histname=0, const char* pdfname=0,bool normalize=false) const ;
  RooHist* pullHist(const char* histname=0, const char* pdfname=0) const { return residHist(histname,pdfname,true); }

protected:

  class DrawOpt {
    public:

    DrawOpt(const char* rawOpt=0) : invisible(kFALSE) { drawOptions[0] = 0 ; initialize(rawOpt) ; }
    void initialize(const char* rawOpt) ;
    const char* rawOpt() const ;

    char drawOptions[128] ;
    Bool_t invisible ;
  } ;


  void initialize();
  TString histName() const ; 
  TString caller(const char *method) const;
  void updateYAxis(Double_t ymin, Double_t ymax, const char *label= "");
  void updateFitRangeNorm(const TH1* hist);
  void updateFitRangeNorm(const RooPlotable* rp, Bool_t refeshNorm=kFALSE);

  RooList _items;            // A list of the items we contain.
  Double_t _padFactor;       // Scale our y-axis to _padFactor of our maximum contents.
  RooAbsRealLValue *_plotVarClone; // A clone of the variable we are plotting.
  RooArgSet *_plotVarSet;    // A list owning the cloned tree nodes of the plotVarClone
  RooArgSet *_normVars;      // Variables that PDF plots should be normalized over

  const RooPlotable* _normObj ;    //! Pointer to normalization object ;
  Double_t _normNumEvts;     // Number of events in histogram (for normalization)
  Double_t _normBinWidth;    // Histogram bin width (for normalization)

  TIterator *_iterator;      //! non-persistent

  Double_t _defYmin ;        // Default minimum for Yaxis (as calculated from contents)
  Double_t _defYmax ;        // Default maximum for Yaxis (as calculated from contents)

  RooPlot(const RooPlot& other); // object cannot be copied

  ClassDef(RooPlot,1)        // Plot frame and container for graphics objects
};

#endif
